const {Usuario} = require("../models");
const bcrypt = require("bcryptjs");
// const jwt = require("jsonwebtoken");
const dotenv = require("dotenv");
const logger = require("../middlewares/logger.middleware");

dotenv.config();

// Autenticación (Login con Nro Documento)
exports.loginNroDocumento = async (req, res) => {
  try {
    const { nrodocumento, contrasenia } = req.params;
    // Verificar si los datos están vacíos
    if (!nrodocumento || !contrasenia) {
      logger.info("NroDocumento y contraseña son obligatorios");
      return res.status(400).json({ status: 'error', error: "NroDocumento y contraseña son obligatorios" });
    }
    // Buscar usuario por correo
    const usuario = await Usuario.findOne({ where: { Documento :nrodocumento } });
    if (!usuario) {
      logger.info("Usuario no encontrado por el correo");
      return res.status(404).json({ status: 'error', error: "Usuario no encontrado con el Nro Documento" });
    }
    // Verificar si la contraseña es correcta
    const esCorrecta = await bcrypt.compare(contrasenia, usuario.Contrasenia);
    if (!esCorrecta) {
      logger.info("Contraseña incorrecta");
      return res.status(401).json({ status: 'error', error: "Contraseña incorrecta" });
    }
    logger.info("Login realizado correctamente");
    res.json({status: 'success',  mensaje: "Login realizado correctamente", data: usuario});
  } catch (error) {
    logger.error("Error al Loguear: " + error.message);
    res.status(500).json({ error: "Error en la autenticación" });
  }
};

// Autenticación (Login con correo)
exports.logincorreo = async (req, res) => {
  try {
    const { correo, contrasenia } = req.params;
    // Verificar si los datos están vacíos
    if (!correo || !contrasenia) {
      logger.info("Correo y contraseña son obligatorios");
      return res.status(400).json({ status: 'error', error: "Correo y contraseña son obligatorios" });
    }
    // Buscar usuario por correo
    const usuario = await Usuario.findOne({ where: { Correo :correo } });
    if (!usuario) {
      logger.info("Usuario no encontrado por el correo");
      return res.status(404).json({ status: 'error', error: "Usuario no encontrado con el Correo" });
    }
    // Verificar si la contraseña es correcta
    const esCorrecta = await bcrypt.compare(contrasenia, usuario.Contrasenia);
    if (!esCorrecta) {
      logger.info("Contraseña incorrecta");
      return res.status(401).json({ status: 'error', error: "Contraseña incorrecta" });
    }
    logger.info("Login realizado correctamente");
    res.json({status: 'success',  mensaje: "Login realizado correctamente", data: usuario});
  } catch (error) {
    logger.error("Error al Loguear: " + error.message);
    res.status(500).json({ error: "Error en la autenticación" });
  }
};

// Autenticación (Login con celular)
exports.logincelular = async (req, res) => {
  try {
    const { celular, contrasenia } = req.params;
    // Verificar si los datos están vacíos
    if (!celular || !contrasenia) {
      logger.info("Correo y contraseña son obligatorios");
      return res.status(400).json({ status: 'error', error: "Correo y contraseña son obligatorios" });
    }
    // Buscar usuario por correo
    const usuario = await Usuario.findOne({ where: { Celular: celular } });
    if (!usuario) {
      logger.info("Usuario no encontrado por el correo");
      return res.status(404).json({ status: 'error', error: "Usuario no encontrado con el Celular" });
    }
    // Verificar si la contraseña es correcta
    const esCorrecta = await bcrypt.compare(contrasenia, usuario.Contrasenia);
    if (!esCorrecta) {
      logger.info("Contraseña incorrecta");
      return res.status(401).json({ status: 'error', error: "Contraseña incorrecta" });
    }
    logger.info("Login realizado correctamente");
    res.json({status: 'success',  mensaje: "Login realizado correctamente", data: usuario});
  } catch (error) {
    console.log(error);
    logger.error("Error al Loguear: " + error.message);
    res.status(500).json({ error: "Error en la autenticación" });
  }
};

// Verificar código
exports.verificarCodigo = async (req, res) => {
  try {
    const { correo, codigoVerificacion } = req.body;
    // Buscar usuario
    const usuario = await Usuario.findOne(
      {
        where: {
          Correo: correo,
          EstadoId: 2 // Asegurarse de que el usuario esté en estado activo
        }
      }
    );
    if (!usuario) return res.status(404).json({ status: 'error', error: "Usuario no encontrado" });
    // Comparar código
    if (usuario.Codigo !== codigoVerificacion) {
      return res.status(400).json({ status: 'error', error: "Código de verificación incorrecto" });
    }
    res.json({ status: 'success', mensaje: "Codigo verificado exitosamente", data: usuario });
  } catch (error) {
    logger.error("Error al verificar codigo: " + error.message);
    res.status(500).json({ error: "Error al verificar el código" });
  }
};

// Actualizar de contraseña con verificación de contraseña actual
exports.updatePassword = async (req, res) => {
  try {
    const { id } = req.params;
    const { contraseniaActual, nuevaContrasenia } = req.body;
    // Verificar si el usuario existe
    const usuario = await Usuario.findByPk(id);
    if (!usuario) return res.status(404).json({ status: 'error', error: "Usuario no encontrado" });
    // Comparar la contraseña actual
    const esCorrecta = await bcrypt.compare(contraseniaActual, usuario.Contrasenia);
    if (!esCorrecta) return res.status(401).json({ status: 'error', error: "Contraseña actual incorrecta" });
    // Encriptar la nueva contraseña
    const hashedPassword = await bcrypt.hash(nuevaContrasenia, 10);
    // Actualizar la contraseña en la base de datos
    await usuario.update({ Contrasenia: hashedPassword });
    logger.info("Contraseña actualizado correctamente con verificación");
    res.json({ status: 'success', mensaje: "Contraseña actualizada correctamente con verificación", data: usuario });
  } catch (error) {
    logger.error("Error al actualizar solo la contrasenia: " + error.message);
    res.status(500).json({ error: "Error al actualizar la contraseña" });
  }
};

// Actualizar de contraseña por parte de un admin
// resetear contraseña de un usuario con su dni o celular
exports.resetPassword = async (req, res) => {
  try {
    const { id } = req.params;
    const { nuevaContrasenia } = req.body;
    // Verificar si el usuario existe
    const usuario = await Usuario.findByPk(id);
    if (!usuario) return res.status(404).json({ status: 'error', error: "Usuario no encontrado" });
    // Comparar la contraseña actual
    const hashedPassword = await bcrypt.hash(nuevaContrasenia, 10);
    // Actualizar la contraseña en la base de datos
    await usuario.update(
      { 
        Contrasenia: hashedPassword,
        EstadoId: 1 // Cambiar el estado a desabilitado
      }
    );
    logger.info("Contraseña actualizar correctamente");
    res.json({ status: 'success', mensaje: "Contraseña actualizar correctamente", data: usuario });
  } catch (error) {
    logger.error("Error al actualizar solo la contrasenia: " + error.message);
    res.status(500).json({ error: "Error al actualizar la contraseña" });
  }
};

