const {Concentimiento, TipoConcentimiento} = require("../models");
const logger = require("../middlewares/logger.middleware");

// Obtener todas las Concentimientoes
exports.getConcentimientoAll = async (req, res) => {
  try {
    const datos = await Concentimiento.findAll(
      {
        include: [
          {
            model: TipoConcentimiento,
            as: 'tipo',
          }
        ]
      }
    );
    logger.info("Concentimientos obtenidos correctamente");
    res.json(datos);
  } catch (error) {
    logger.error("Error al obtener Concentimientos: " + error.message);
    res.status(500).json({ error: "Error al obtener Concentimientoes" });
  }
};

// Obtener Concentimiento por Historia Clinica
exports.getConcentimientoByPaciente = async (req, res) => {
  try {
    const { paciente } = req.params;
    const datos = await Concentimiento.findAll(
      {
        where: {
          PacienteId: paciente
        },
        include: [
          {
            model: TipoConcentimiento,
            as: 'tipo',
          }
        ]
      }
    );
    if (!datos) {
      logger.error("Concentimiento no encontrada por su asociacion con Historia Clinica");
      return res.status(404).json({ error: "Concentimientos no encontrada" });
    }
    logger.info("Concentimientos obtenidas correctamente por su asociacion con la Historia Clinica");
    res.json(datos);
  } catch (error) {
    logger.error("Error al obtener el Concentimiento mediante por Historia Clinica: " + error.message);
    res.status(500).json({ error: "Error al obtener Concentimientos" });
  }
};

// Obtener una Concentimiento por ID
exports.getConcentimientoById = async (req, res) => {
  try {
    const dato = await Concentimiento.findByPk(
      req.params.id,
      {
        include: [
          {
            model: TipoConcentimiento,
            as: 'tipo',
          }
        ]
      }
    );
    if (!dato) {
      logger.error("Concentimiento no encontrada por su ID");
      return res.status(404).json({ error: "Concentimiento no encontrada" });
    }
    logger.info("Concentimiento obtenido correctamente por su ID");
    res.json(dato);
  } catch (error) {
    logger.error("Error al obtener el Concentimiento mediante su Id: " + error.message);
    res.status(500).json({ error: "Error al obtener Concentimiento" });
  }
};

// Crear una Concentimiento
exports.createConcentimiento = async (req, res) => {
  try {
    const dato = await Concentimiento.create(req.body);
    logger.info("Concentimientos creado correctamente");
    res.status(201).json(dato);
  } catch (error) {
    logger.error("Error al crear un Concentimiento: " + error.message);
    res.status(500).json({ error: "Error al crear Concentimiento" });
  }
};

// Actualizar una Concentimiento
exports.updateConcentimiento = async (req, res) => {
  try {
    const dato = await Concentimiento.findByPk(req.params.id);
    if (!dato) {
      logger.error("Concentimiento no encontrada por su ID");
      return res.status(404).json({ error: "Concentimiento no encontrada" });
    }
    await dato.update(req.body);
    logger.info("Concentimientos actualizado correctamente");
    res.json(dato);
  } catch (error) {
    logger.error("Error al actualizar un Concentimiento: " + error.message);
    res.status(500).json({ error: "Error al actualizar Concentimiento" });
  }
};

// Eliminar una Concentimiento
exports.deleteConcentimiento = async (req, res) => {
  try {
    const dato = await Concentimiento.findByPk(req.params.id);
    if (!dato) {
      logger.error("Concentimiento no encontrada por su ID");
      return res.status(404).json({ error: "Concentimiento no encontrada" });
    }
    await Concentimiento.destroy();
    logger.info("Concentimientos eliminado correctamente");
    res.json({ message: "Concentimiento eliminada" });
  } catch (error) {
    logger.error("Error al elimininar un Concentimiento: " + error.message);
    res.status(500).json({ error: "Error al eliminar Concentimiento" });
  }
};

