const {Documento, TipoDocPDF} = require("../models");
const logger = require("../middlewares/logger.middleware");

// Obtener todas los Documentos
exports.getDocumentoAll = async (req, res) => {
  try {
    const documentos = await Documento.findAll(
      {
        include: [
          {
            model: TipoDocPDF,
            as: 'tipo'
          }
        ],
        order: [['TipoDocPDFId', 'ASC']] // Ordenar por TipoDocPDFId en orden ascendente
      }
    );
    logger.info("Documentos obtenidos correctamente");
    res.json(documentos);
  } catch (error) {
    logger.error("Error al obtener Documentos: " + error.message);
    res.status(500).json({ error: "Error al obtener Documentoes" });
  }
};

// Obtener todas los Documentos por trabajador
exports.getDocumentoByTrabajador = async (req, res) => {
  try {
    const { trabajador } = req.params;
    const documentos = await Documento.findAll(
      {
        where: {
          UsuarioId: trabajador
        },
        include: [
          {
            model: TipoDocPDF,
            as: 'tipo'
          }
        ],
        order: [['TipoDocPDFId', 'ASC']] // Ordenar por fecharegistro en orden descendente
      }
    );
    logger.info("Documentos obtenidos correctamente");
    res.json(documentos);
  } catch (error) {
    logger.error("Error al obtener Documentos: " + error.message);
    res.status(500).json({ error: "Error al obtener Documentoes" });
  }
};

// Obtener una Documento por ID
exports.getDocumentoById = async (req, res) => {
  try {
    const documento = await Documento.findByPk(
      {
        where: { id: req.params.id },
        include: [
          {
            model: TipoDocPDF,
            as: 'tipo'
          }
        ]
      }
    );
    if (!documento) {
      logger.error("Documento no encontrada por su ID");
      return res.status(404).json({ error: "Documento no encontrada" });
    }
    logger.info("Documento obtenido correctamente por su ID");
    res.json(documento);
  } catch (error) {
    logger.error("Error al obtener el Documento mediante su Id: " + error.message);
    res.status(500).json({ error: "Error al obtener Documento" });
  }
};

// Crear una Documento
exports.createDocumento = async (req, res) => {
  try {
    const documento = await Documento.create(req.body);
    logger.info("Documentos creado correctamente");
    res.status(201).json(documento);
  } catch (error) {
    logger.error("Error al crear un Documento: " + error.message);
    res.status(500).json({ error: "Error al crear Documento" });
  }
};

// Actualizar una Documento
exports.updateDocumento = async (req, res) => {
  try {
    const documento = await Documento.findByPk(req.params.id);
    if (!documento) {
      logger.error("Documento no encontrada por su ID");
      return res.status(404).json({ error: "Documento no encontrada" });
    }
    await documento.update(req.body);
    logger.info("Documentos actualizado correctamente");
    res.json(documento);
  } catch (error) {
    logger.error("Error al actualizar un Documento: " + error.message);
    res.status(500).json({ error: "Error al actualizar Documento" });
  }
};

// Eliminar una Documento
exports.deleteDocumento = async (req, res) => {
  try {
    const documento = await Documento.findByPk(req.params.id);
    if (!documento) {
      logger.error("Documento no encontrada por su ID");
      return res.status(404).json({ error: "Documento no encontrada" });
    }
    await Documento.destroy();
    logger.info("Documentos eliminado correctamente");
    res.json({ message: "Documento eliminada" });
  } catch (error) {
    logger.error("Error al elimininar un Documento: " + error.message);
    res.status(500).json({ error: "Error al eliminar Documento" });
  }
};

