const {EnfermedadActual} = require("../models");
const logger = require("../middlewares/logger.middleware");

// Obtener todas las EnfermedadActuales
exports.getEnfermedadActualAll = async (req, res) => {
  try {
    const datos = await EnfermedadActual.findAll();
    logger.info("EnfermedadActuals obtenidos correctamente");
    res.json(datos);
  } catch (error) {
    logger.error("Error al obtener EnfermedadActuals: " + error.message);
    res.status(500).json({ error: "Error al obtener EnfermedadActuales" });
  }
};

// Obtener EnfermedadActual por Historia Clinica
exports.getEnfermedadActualByPaciente = async (req, res) => {
  try {
    const { paciente } = req.params;
    const datos = await EnfermedadActual.findAll(
      {
        where: {
          PacienteId: paciente
        }
      }
    );
    if (!datos) {
      logger.error("EnfermedadActual no encontrada por su asociacion con Historia Clinica");
      return res.status(404).json({ error: "EnfermedadActuals no encontrada" });
    }
    logger.info("EnfermedadActuals obtenidas correctamente por su asociacion con la Historia Clinica");
    res.json(datos);
  } catch (error) {
    logger.error("Error al obtener el EnfermedadActual mediante por Historia Clinica: " + error.message);
    res.status(500).json({ error: "Error al obtener EnfermedadActuals" });
  }
};

// Obtener una EnfermedadActual por ID
exports.getEnfermedadActualById = async (req, res) => {
  try {
    const dato = await EnfermedadActual.findByPk(req.params.id);
    if (!dato) {
      logger.error("EnfermedadActual no encontrada por su ID");
      return res.status(404).json({ error: "EnfermedadActual no encontrada" });
    }
    logger.info("EnfermedadActual obtenido correctamente por su ID");
    res.json(dato);
  } catch (error) {
    logger.error("Error al obtener el EnfermedadActual mediante su Id: " + error.message);
    res.status(500).json({ error: "Error al obtener EnfermedadActual" });
  }
};

// Crear una EnfermedadActual
exports.createEnfermedadActual = async (req, res) => {
  try {
    const dato = await EnfermedadActual.create(req.body);
    logger.info("EnfermedadActuals creado correctamente");
    res.status(201).json(dato);
  } catch (error) {
    logger.error("Error al crear un EnfermedadActual: " + error.message);
    res.status(500).json({ error: "Error al crear EnfermedadActual" });
  }
};

// Actualizar una EnfermedadActual
exports.updateEnfermedadActual = async (req, res) => {
  try {
    const dato = await EnfermedadActual.findByPk(req.params.id);
    if (!dato) {
      logger.error("EnfermedadActual no encontrada por su ID");
      return res.status(404).json({ error: "EnfermedadActual no encontrada" });
    }
    await dato.update(req.body);
    logger.info("EnfermedadActuals actualizado correctamente");
    res.json(dato);
  } catch (error) {
    logger.error("Error al actualizar un EnfermedadActual: " + error.message);
    res.status(500).json({ error: "Error al actualizar EnfermedadActual" });
  }
};

// Eliminar una EnfermedadActual
exports.deleteEnfermedadActual = async (req, res) => {
  try {
    const dato = await EnfermedadActual.findByPk(req.params.id);
    if (!dato) {
      logger.error("EnfermedadActual no encontrada por su ID");
      return res.status(404).json({ error: "EnfermedadActual no encontrada" });
    }
    await EnfermedadActual.destroy();
    logger.info("EnfermedadActuals eliminado correctamente");
    res.json({ message: "EnfermedadActual eliminada" });
  } catch (error) {
    logger.error("Error al elimininar un EnfermedadActual: " + error.message);
    res.status(500).json({ error: "Error al eliminar EnfermedadActual" });
  }
};

