const {Evolucion} = require("../models");
const logger = require("../middlewares/logger.middleware");

// Obtener todas las Evoluciones
exports.getEvolucionAll = async (req, res) => {
  try {
    const datos = await Evolucion.findAll();
    logger.info("Evolucions obtenidos correctamente");
    res.json(datos);
  } catch (error) {
    logger.error("Error al obtener Evolucions: " + error.message);
    res.status(500).json({ error: "Error al obtener Evoluciones" });
  }
};

// Obtener Evolucion por Historia Clinica
exports.getEvolucionByPaciente = async (req, res) => {
  try {
    const { paciente } = req.params;
    const datos = await Evolucion.findAll(
      {
        where: {
          PacienteId: paciente
        }
      }
    );
    if (!datos) {
      logger.error("Evolucion no encontrada por su asociacion con Historia Clinica");
      return res.status(404).json({ error: "Evolucions no encontrada" });
    }
    logger.info("Evolucions obtenidas correctamente por su asociacion con la Historia Clinica");
    res.json(datos);
  } catch (error) {
    logger.error("Error al obtener el Evolucion mediante por Historia Clinica: " + error.message);
    res.status(500).json({ error: "Error al obtener Evolucions" });
  }
};

// Obtener una Evolucion por ID
exports.getEvolucionById = async (req, res) => {
  try {
    const dato = await Evolucion.findByPk(req.params.id);
    if (!dato) {
      logger.error("Evolucion no encontrada por su ID");
      return res.status(404).json({ error: "Evolucion no encontrada" });
    }
    logger.info("Evolucion obtenido correctamente por su ID");
    res.json(dato);
  } catch (error) {
    logger.error("Error al obtener el Evolucion mediante su Id: " + error.message);
    res.status(500).json({ error: "Error al obtener Evolucion" });
  }
};

// Crear una Evolucion
exports.createEvolucion = async (req, res) => {
  try {
    const dato = await Evolucion.create(req.body);
    logger.info("Evolucions creado correctamente");
    res.status(201).json(dato);
  } catch (error) {
    logger.error("Error al crear un Evolucion: " + error.message);
    res.status(500).json({ error: "Error al crear Evolucion" });
  }
};

// Actualizar una Evolucion
exports.updateEvolucion = async (req, res) => {
  try {
    const dato = await Evolucion.findByPk(req.params.id);
    if (!dato) {
      logger.error("Evolucion no encontrada por su ID");
      return res.status(404).json({ error: "Evolucion no encontrada" });
    }
    await dato.update(req.body);
    logger.info("Evolucions actualizado correctamente");
    res.json(dato);
  } catch (error) {
    logger.error("Error al actualizar un Evolucion: " + error.message);
    res.status(500).json({ error: "Error al actualizar Evolucion" });
  }
};

// Eliminar una Evolucion
exports.deleteEvolucion = async (req, res) => {
  try {
    const dato = await Evolucion.findByPk(req.params.id);
    if (!dato) {
      logger.error("Evolucion no encontrada por su ID");
      return res.status(404).json({ error: "Evolucion no encontrada" });
    }
    await Evolucion.destroy();
    logger.info("Evolucions eliminado correctamente");
    res.json({ message: "Evolucion eliminada" });
  } catch (error) {
    logger.error("Error al elimininar un Evolucion: " + error.message);
    res.status(500).json({ error: "Error al eliminar Evolucion" });
  }
};

