const {ExploracionFisica} = require("../models");
const logger = require("../middlewares/logger.middleware");

// Obtener todas las ExploracionFisicaes
exports.getExploracionFisicaAll = async (req, res) => {
  try {
    const datos = await ExploracionFisica.findAll();
    logger.info("ExploracionFisicas obtenidos correctamente");
    res.json(datos);
  } catch (error) {
    logger.error("Error al obtener ExploracionFisicas: " + error.message);
    res.status(500).json({ error: "Error al obtener ExploracionFisicaes" });
  }
};

// Obtener ExploracionFisica por Historia Clinica
exports.getExploracionFisicaByPaciente = async (req, res) => {
  try {
    const { paciente } = req.params;
    const datos = await ExploracionFisica.findAll(
      {
        where: {
          PacienteId: paciente
        }
      }
    );
    if (!datos) {
      logger.error("ExploracionFisica no encontrada por su asociacion con Historia Clinica");
      return res.status(404).json({ error: "ExploracionFisicas no encontrada" });
    }
    logger.info("ExploracionFisicas obtenidas correctamente por su asociacion con la Historia Clinica");
    res.json(datos);
  } catch (error) {
    logger.error("Error al obtener el ExploracionFisica mediante por Historia Clinica: " + error.message);
    res.status(500).json({ error: "Error al obtener ExploracionFisicas" });
  }
};

// Obtener una ExploracionFisica por ID
exports.getExploracionFisicaById = async (req, res) => {
  try {
    const dato = await ExploracionFisica.findByPk(req.params.id);
    if (!dato) {
      logger.error("ExploracionFisica no encontrada por su ID");
      return res.status(404).json({ error: "ExploracionFisica no encontrada" });
    }
    logger.info("ExploracionFisica obtenido correctamente por su ID");
    res.json(dato);
  } catch (error) {
    logger.error("Error al obtener el ExploracionFisica mediante su Id: " + error.message);
    res.status(500).json({ error: "Error al obtener ExploracionFisica" });
  }
};

// Crear una ExploracionFisica
exports.createExploracionFisica = async (req, res) => {
  try {
    const dato = await ExploracionFisica.create(req.body);
    logger.info("ExploracionFisicas creado correctamente");
    res.status(201).json(dato);
  } catch (error) {
    logger.error("Error al crear un ExploracionFisica: " + error.message);
    res.status(500).json({ error: "Error al crear ExploracionFisica" });
  }
};

// Actualizar una ExploracionFisica
exports.updateExploracionFisica = async (req, res) => {
  try {
    const dato = await ExploracionFisica.findByPk(req.params.id);
    if (!dato) {
      logger.error("ExploracionFisica no encontrada por su ID");
      return res.status(404).json({ error: "ExploracionFisica no encontrada" });
    }
    await dato.update(req.body);
    logger.info("ExploracionFisicas actualizado correctamente");
    res.json(dato);
  } catch (error) {
    logger.error("Error al actualizar un ExploracionFisica: " + error.message);
    res.status(500).json({ error: "Error al actualizar ExploracionFisica" });
  }
};

// Eliminar una ExploracionFisica
exports.deleteExploracionFisica = async (req, res) => {
  try {
    const dato = await ExploracionFisica.findByPk(req.params.id);
    if (!dato) {
      logger.error("ExploracionFisica no encontrada por su ID");
      return res.status(404).json({ error: "ExploracionFisica no encontrada" });
    }
    await ExploracionFisica.destroy();
    logger.info("ExploracionFisicas eliminado correctamente");
    res.json({ message: "ExploracionFisica eliminada" });
  } catch (error) {
    logger.error("Error al elimininar un ExploracionFisica: " + error.message);
    res.status(500).json({ error: "Error al eliminar ExploracionFisica" });
  }
};

