const {Ingreso} = require("../models");
const logger = require("../middlewares/logger.middleware");

// Obtener todas las Ingresoes
exports.getIngresoAll = async (req, res) => {
  try {
    const ingresos = await Ingreso.findAll();
    logger.info("Ingresos obtenidos correctamente");
    res.json(ingresos);
  } catch (error) {
    logger.error("Error al obtener Ingresos: " + error.message);
    res.status(500).json({ error: "Error al obtener Ingresoes" });
  }
};

// Filtrar Ingresos por sede
exports.getIngresosBySede = async (req, res) => {
  try {
    const { sede } = req.params;
    const ingresos = await Ingreso.findAll(
      {
        where: {
          SedeId: sede
        },
        order: [
          ['FechaRegistro', 'DESC'] // Ordena de forma ascendente por Fecha
        ]
      }
    );
    res.json(ingresos);
  } catch (error) {
    res.status(500).json(
      {
        error: "Error al filtrar Ingresoes por sede"
      }
    );
  }
};

// Filtrar Ingresoes por sede y tipo pago
exports.getIngresoBySedeandTipoPago = async (req, res) => {
  try {
    const { sede, tipopago } = req.params;
    const ingresos = await Ingreso.findAll(
      {
        where: {
          SedeId: sede,
          TipoPagoId : tipopago
        },
        order: [
          ['FechaRegistro', 'DESC'] // Ordena de forma ascendente por Fecha
        ]
      }
    );
    res.json(ingresos);
  } catch (error) {
    res.status(500).json(
      {
        error: "Error al filtrar Ingresos por sede"
      }
    );
  }
};

// Filtrar Ingresoes por sede y tipo ingreso
exports.getIngresoBySedeandTipoIngreso = async (req, res) => {
  try {
    const { sede, tipoingreso } = req.params;
    const ingresos = await Ingreso.findAll(
      {
        where: {
          SedeId: sede,
          TipoIngresoId : tipoingreso
        },
        order: [
          ['FechaRegistro', 'DESC'] // Ordena de forma ascendente por Fecha
        ]
      }
    );
    res.json(ingresos);
  } catch (error) {
    res.status(500).json(
      {
        error: "Error al filtrar Ingresos por sede"
      }
    );
  }
};

// Filtrar Ingresoes por sede y tratamiento
exports.getIngresoBySedeandTipoProveedor = async (req, res) => {
  try {
    const { sede, tipoproveedor } = req.params;
    const ingresos = await Ingreso.findAll(
      {
        where: {
          SedeId: sede,
          TipoProveedorId : tipoproveedor
        },
        order: [
          ['FechaRegistro', 'DESC'] // Ordena de forma ascendente por Fecha
        ]
      }
    );
    res.json(ingresos);
  } catch (error) {
    res.status(500).json(
      {
        error: "Error al filtrar Ingresos por sede"
      }
    );
  }
};

// Filtrar Ingresoes por sede y tratamiento
exports.getIngresoByReport = async (req, res) => {
  try {
    const { fechainicio, fechafin } = req.params;
    const ingresos = await Ingreso.findAll(
      {
        attributes: [
          'SedeId',
          'TipoIngresoId',
          'TipoPagoId',
          [fn('SUM', col('MontoPagado')), 'TotalMontoPagado']
        ],
        where: {
          FechaRegistro: {
            [Op.between]: [fechainicio, fechafin]
          }
        },
        group: ['SedeId', 'TipoIngresoId', 'TipoPagoId'],
        order: [
          ['SedeId', 'DESC'],
          ['TipoIngresoId', 'ASC'],
          ['TipoPagoId', 'ASC'] 
        ]
      }
    );
    res.json(ingresos);
  } catch (error) {
    res.status(500).json(
      {
        error: "Error al recibir el reporte"
      }
    );
  }
};

// Obtener una Ingreso por ID
exports.getIngresoById = async (req, res) => {
  try {
    const ingresos = await Ingreso.findByPk(req.params.id);
    if (!ingresos) {
      logger.error("Ingreso no encontrada por su ID");
      return res.status(404).json({ error: "Ingreso no encontrada" });
    }
    logger.info("Ingreso obtenido correctamente por su ID");
    res.json(ingresos);
  } catch (error) {
    logger.error("Error al obtener el Ingreso mediante su Id: " + error.message);
    res.status(500).json({ error: "Error al obtener Ingreso" });
  }
};

// Crear una Ingreso
exports.createIngreso = async (req, res) => {
  try {
    const ingresos = await Ingreso.create(req.body);
    logger.info("Ingresos creado correctamente");
    res.status(201).json(ingresos);
  } catch (error) {
    logger.error("Error al crear un Ingreso: " + error.message);
    res.status(500).json({ error: "Error al crear Ingreso" });
  }
};

// Actualizar una Ingreso
exports.updateIngreso = async (req, res) => {
  try {
    const ingreso = await Ingreso.findByPk(req.params.id);
    if (!ingreso) {
      logger.error("Ingreso no encontrada por su ID");
      return res.status(404).json({ error: "Ingreso no encontrada" });
    }
    await ingreso.update(req.body);
    logger.info("Ingresos actualizado correctamente");
    res.json(ingreso);
  } catch (error) {
    logger.error("Error al actualizar un Ingreso: " + error.message);
    res.status(500).json({ error: "Error al actualizar Ingreso" });
  }
};

// Eliminar una Ingreso
exports.deleteIngreso = async (req, res) => {
  try {
    const ingresos = await Ingreso.findByPk(req.params.id);
    if (!ingresos) {
      logger.error("Ingreso no encontrada por su ID");
      return res.status(404).json({ error: "Ingreso no encontrada" });
    }
    await Ingreso.destroy();
    logger.info("Ingresos eliminado correctamente");
    res.json({ message: "Ingreso eliminada" });
  } catch (error) {
    logger.error("Error al elimininar un Ingreso: " + error.message);
    res.status(500).json({ error: "Error al eliminar Ingreso" });
  }
};
