const {Inventario, Almacen, Insumo} = require("../models");
const logger = require("../middlewares/logger.middleware");

// Obtener todas las Inventarioes
exports.getInventarioAll = async (req, res) => {
  try {
    const inventario = await Inventario.findAll(
      {
        include: [
          {
            model: Almacen,
            as: "almacen"
          },
          {
            model: Insumo,
            as: "insumo"
          }
        ],
        order: [['createdAt', 'DESC']]
      }
    );
    logger.info("Inventarios obtenidos correctamente");
    res.json(inventario);
  } catch (error) {
    logger.error("Error al obtener Inventarios: " + error.message);
    res.status(500).json({ error: "Error al obtener Inventarioes" });
  }
};

// Obtener el inventario en tal sede
exports.getInventarioBySede = async (req, res) => {
  try {
    const { sede } = req.params;
    const inventarios = await Inventario.findAll(
      {
        include: [
          {
            model: Almacen,
            as: "almacen",
            where: {
              SedeId: sede
            }
          },
          {
            model: Insumo,
            as: "insumo"
          }
        ]
      }
    );
    logger.info("Inventarios obtenidos correctamente");
    res.json(inventarios);
  } catch (error) {
    logger.error("Error al obtener Inventarios: " + error.message);
    res.status(500).json({ error: "Error al obtener Inventarioes" });
  }
};

// Obtener todas las Inventarioes de tal almacen
exports.getInventarioByAlmacen = async (req, res) => {
  try {
    const { almacen } = req.params;
    const inventarios = await Inventario.findAll(
      {
        where: {
          AlmacenId: almacen
        },
        include: [
          {
            model: Insumo,
            as: "insumo"
          },
          {
            model: Almacen,
            as: "almacen"
          }
        ]
      }
    );
    logger.info("Inventarios obtenidos correctamente");
    res.json(inventarios);
  } catch (error) {
    logger.error("Error al obtener Inventarios: " + error.message);
    res.status(500).json({ error: "Error al obtener Inventarioes" });
  }
};

// Obtener el inventario en tal sede
exports.getInventarioBySedeeInsumo = async (req, res) => {
  try {
    const { insumo, sede } = req.params;
    const inventarios = await Inventario.findAll(
      {
        where: {
          InsumoId: insumo
        },
        include: [
          {
            model: Almacen,
            as: "almacen",
            where: {
              SedeId: sede
            }
          },
          {
            model: Insumo,
            as: "insumo"
          }
        ]
      }
    );
    logger.info("Inventarios obtenidos correctamente");
    res.json(inventarios);
  } catch (error) {
    logger.error("Error al obtener Inventarios: " + error.message);
    res.status(500).json({ error: "Error al obtener Inventarioes" });
  }
};

// Obtener el inventario de tal insumo en un almacen especifico
exports.getInventarioByAlmacenANDInsumo = async (req, res) => {
  try {
    const { almacen, insumo } = req.params;
    const inventarios = await Inventario.findOne(
      {
        where: {
          AlmacenId: almacen,
          InsumoId: insumo
        },
        include: [
          {
            model: Almacen,
            as: "almacen"
          },
          {
            model: Insumo,
            as: "insumo"
          }
        ]
      }
    );
    logger.info("Inventarios obtenidos correctamente");
    res.json(inventarios);
  } catch (error) {
    logger.error("Error al obtener Inventarios: " + error.message);
    res.status(500).json({ error: "Error al obtener Inventarioes" });
  }
};

// Obtener una Inventario por ID
exports.getInventarioById = async (req, res) => {
  try {
    const inventario = await Inventario.findByPk(req.params.id);
    if (!inventario) {
      logger.error("Inventario no encontrada por su ID");
      return res.status(404).json({ error: "Inventario no encontrada" });
    }
    logger.info("Inventario obtenido correctamente por su ID");
    res.json(inventario);
  } catch (error) {
    logger.error("Error al obtener el Inventario mediante su Id: " + error.message);
    res.status(500).json({ error: "Error al obtener Inventario" });
  }
};

// Crear una Inventario
exports.createInventario = async (req, res) => {
  try {
    const inventario = await Inventario.create(req.body);
    logger.info("Inventarios creado correctamente");
    res.status(201).json(inventario);
  } catch (error) {
    logger.error("Error al crear un Inventario: " + error.message);
    res.status(500).json({ error: "Error al crear Inventario" });
  }
};

// Actualizar una Inventario
exports.updateInventario = async (req, res) => {
  try {
    const inventario = await Inventario.findByPk(req.params.id);
    if (!inventario) {
      logger.error("Inventario no encontrada por su ID");
      return res.status(404).json({ error: "Inventario no encontrada" });
    }
    await inventario.update(req.body);
    logger.info("Inventarios actualizado correctamente");
    res.json(inventario);
  } catch (error) {
    logger.error("Error al actualizar un Inventario: " + error.message);
    res.status(500).json({ error: "Error al actualizar Inventario" });
  }
};

// Eliminar una Inventario
exports.deleteInventario = async (req, res) => {
  try {
    const inventario = await Inventario.findByPk(req.params.id);
    if (!inventario) {
      logger.error("Inventario no encontrada por su ID");
      return res.status(404).json({ error: "Inventario no encontrada" });
    }
    await Inventario.destroy();
    logger.info("Inventarios eliminado correctamente");
    res.json({ message: "Inventario eliminada" });
  } catch (error) {
    logger.error("Error al elimininar un Inventario: " + error.message);
    res.status(500).json({ error: "Error al eliminar Inventario" });
  }
};

