const {Kardex, Almacen, Insumo} = require("../models");
const logger = require("../middlewares/logger.middleware");

// Obtener todas los Kardexes
exports.getKardexAll = async (req, res) => {
  try {
    const datos = await Kardex.findAll(
      {
        include: [
          {
            model: Almacen,
            as: "almacen"
          },
          {
            model: Insumo,
            as: "insumo"
          }
        ],
        order: [['createdAt', 'DESC']]
      }
    );
    logger.info("Kardexs obtenidos correctamente");
    res.json(datos);
  } catch (error) {
    logger.error("Error al obtener Kardexs: " + error.message);
    res.status(500).json({ error: "Error al obtener Kardexes" });
  }
};

// Obtener todas los movimientos desde el subalmacen
exports.getKardexByAlmacen = async (req, res) => {
  try {
    const { almacen } = req.params;
    const datos = await Kardex.findAll(
      {
        where: {
          AlmacenId: almacen
        },
        include: [
          {
            model: Almacen,
            as: "almacen"
          },
          {
            model: Insumo,
            as: "insumo"
          }
        ]
      }
    );
    logger.info("Kardexs obtenidos correctamente");
    res.json(datos);
  } catch (error) {
    logger.error("Error al obtener Kardexs: " + error.message);
    res.status(500).json({ error: "Error al obtener Kardexes" });
  }
};

// Obtener todas los movimientos de tal insumo
exports.getKardexByInsumo = async (req, res) => {
  try {
    const { insumo } = req.params;
    const datos = await Kardex.findAll(
      {
        where: {
          InsumoId: insumo
        },
        include: [
          {
            model: Almacen,
            as: "almacen"
          },
          {
            model: Insumo,
            as: "insumo"
          }
        ]
      }
    );
    logger.info("Kardexs obtenidos correctamente");
    res.json(datos);
  } catch (error) {
    logger.error("Error al obtener Kardexs: " + error.message);
    res.status(500).json({ error: "Error al obtener Kardexes" });
  }
};

// Obtener todas los movimientos de tal insumo en el almacen
exports.getKardexByMovimientoInsumo = async (req, res) => {
  try {
    const { insumo, almacen } = req.params;
    const datos = await Kardex.findAll(
      {
        where: {
          AlmacenId: almacen,
          InsumoId: insumo
        },
        include: [
          {
            model: Almacen,
            as: "almacen"
          },
          {
            model: Insumo,
            as: "insumo"
          }
        ]
      }
    );
    logger.info("Kardexs obtenidos correctamente");
    res.json(datos);
  } catch (error) {
    logger.error("Error al obtener Kardexs: " + error.message);
    res.status(500).json({ error: "Error al obtener Kardexes" });
  }
};

// Obtener una Kardex por ID
exports.getKardexById = async (req, res) => {
  try {
    const dato = await Kardex.findByPk(req.params.id);
    if (!dato) {
      logger.error("Kardex no encontrada por su ID");
      return res.status(404).json({ error: "Kardex no encontrada" });
    }
    logger.info("Kardex obtenido correctamente por su ID");
    res.json(dato);
  } catch (error) {
    logger.error("Error al obtener el Kardex mediante su Id: " + error.message);
    res.status(500).json({ error: "Error al obtener Kardex" });
  }
};

// Crear una Kardex
exports.createKardex = async (req, res) => {
  try {
    const dato = await Kardex.create(req.body);
    logger.info("Kardexs creado correctamente");
    res.status(201).json(dato);
  } catch (error) {
    logger.error("Error al crear un Kardex: " + error.message);
    res.status(500).json({ error: "Error al crear Kardex" });
  }
};

// Actualizar una Kardex
exports.updateKardex = async (req, res) => {
  try {
    const dato = await Kardex.findByPk(req.params.id);
    if (!dato) {
      logger.error("Kardex no encontrada por su ID");
      return res.status(404).json({ error: "Kardex no encontrada" });
    }
    await dato.update(req.body);
    logger.info("Kardexs actualizado correctamente");
    res.json(dato);
  } catch (error) {
    logger.error("Error al actualizar un Kardex: " + error.message);
    res.status(500).json({ error: "Error al actualizar Kardex" });
  }
};

// Eliminar una Kardex
exports.deleteKardex = async (req, res) => {
  try {
    const dato = await Kardex.findByPk(req.params.id);
    if (!dato) {
      logger.error("Kardex no encontrada por su ID");
      return res.status(404).json({ error: "Kardex no encontrada" });
    }
    await Kardex.destroy();
    logger.info("Kardexs eliminado correctamente");
    res.json({ message: "Kardex eliminada" });
  } catch (error) {
    logger.error("Error al elimininar un Kardex: " + error.message);
    res.status(500).json({ error: "Error al eliminar Kardex" });
  }
};

