const {Movimiento, Patrimonio} = require("../models");
const logger = require("../middlewares/logger.middleware");

// Obtener todas los Movimientos
exports.getMovimientoAll = async (req, res) => {
  try {
    const datos = await Movimiento.findAll(
      {
        include: [
          {
            model: Patrimonio,
            as: "patrimonio"
          }
        ]
      }
    );
    logger.info("Movimientos obtenidos correctamente");
    res.json(datos);
  } catch (error) {
    logger.error("Error al obtener Movimientos: " + error.message);
    res.status(500).json({ error: "Error al obtener Movimientos" });
  }
};

// Obtener todas los movimientos desde el subalmacen
exports.getMovimientoBySede = async (req, res) => {
  try {
    const { sede } = req.params;
    const datos = await Movimiento.findAll(
      {
        where: {
          SedeId: sede
        },
        include: [
          {
            model: Patrimonio,
            as: "patrimonio"
          }
        ]
      }
    );
    logger.info("Movimientos obtenidos correctamente");
    res.json(datos);
  } catch (error) {
    logger.error("Error al obtener Movimientos: " + error.message);
    res.status(500).json({ error: "Error al obtener Movimientos" });
  }
};

// Obtener todas los movimientos de tal insumo
exports.getMovimientoByPatrimonio = async (req, res) => {
  try {
    const { patrimonio } = req.params;
    const datos = await Movimiento.findAll(
      {
        where: {
          PatrimonioId: patrimonio
        }
      }
    );
    logger.info("Movimientos obtenidos correctamente");
    res.json(datos);
  } catch (error) {
    logger.error("Error al obtener Movimientos: " + error.message);
    res.status(500).json({ error: "Error al obtener Movimientos" });
  }
};

// Obtener una Movimiento por ID
exports.getMovimientoById = async (req, res) => {
  try {
    const dato = await Movimiento.findByPk(req.params.id);
    if (!dato) {
      logger.error("Movimiento no encontrada por su ID");
      return res.status(404).json({ error: "Movimiento no encontrada" });
    }
    logger.info("Movimiento obtenido correctamente por su ID");
    res.json(dato);
  } catch (error) {
    logger.error("Error al obtener el Movimiento mediante su Id: " + error.message);
    res.status(500).json({ error: "Error al obtener Movimiento" });
  }
};

// Crear una Movimiento
exports.createMovimiento = async (req, res) => {
  try {
    const dato = await Movimiento.create(req.body);
    logger.info("Movimientos creado correctamente");
    res.status(201).json(dato);
  } catch (error) {
    logger.error("Error al crear un Movimiento: " + error.message);
    res.status(500).json({ error: "Error al crear Movimiento" });
  }
};

// Actualizar una Movimiento
exports.updateMovimiento = async (req, res) => {
  try {
    const dato = await Movimiento.findByPk(req.params.id);
    if (!dato) {
      logger.error("Movimiento no encontrada por su ID");
      return res.status(404).json({ error: "Movimiento no encontrada" });
    }
    await dato.update(req.body);
    logger.info("Movimientos actualizado correctamente");
    res.json(dato);
  } catch (error) {
    logger.error("Error al actualizar un Movimiento: " + error.message);
    res.status(500).json({ error: "Error al actualizar Movimiento" });
  }
};

// Eliminar una Movimiento
exports.deleteMovimiento = async (req, res) => {
  try {
    const dato = await Movimiento.findByPk(req.params.id);
    if (!dato) {
      logger.error("Movimiento no encontrada por su ID");
      return res.status(404).json({ error: "Movimiento no encontrada" });
    }
    await dato.destroy();
    logger.info("Movimientos eliminado correctamente");
    res.json({ message: "Movimiento eliminada" });
  } catch (error) {
    logger.error("Error al elimininar un Movimiento: " + error.message);
    res.status(500).json({ error: "Error al eliminar Movimiento" });
  }
};

