const { Op, Sequelize } = require('sequelize');
const {Paciente} = require("../models");
const logger = require("../middlewares/logger.middleware");

// Obtener todos los pacientes
exports.getPacienteAll = async (req, res) => {
  try {
    const pacientes = await Paciente.findAll();
    logger.info("Pacientes obtenidos correctamente");
    res.json(pacientes);
  } catch (error) {
    logger.error("Error al obtener Pacientes: " + error.message);
    res.status(500).json({ error: "Error al obtener pacientes" });
  }
};

// Obtener un paciente por ID
exports.getPacienteById = async (req, res) => {
  try {
    const paciente = await Paciente.findByPk(req.params.id);
    if (!paciente) {
      logger.error("Paciente no encontrado por su ID");
      return res.status(404).json({ error: "Paciente no encontrado" });
    }
    logger.info("Paciente obtenido correctamente por su ID");
    res.json(paciente);
  } catch (error) {
    logger.error("Error al obtener el Paciente mediante su Id: " + error.message);
    res.status(500).json({ error: "Error al obtener paciente" });
  }
};

// Crear un paciente
exports.createPaciente = async (req, res) => {
  try {
    const paciente = await Paciente.create(req.body);
    logger.info("Pacientes creado correctamente");
    res.status(201).json(paciente);
  } catch (error) {
    console.log(error);
    logger.error("Error al crear un Paciente: " + error.message);
    res.status(500).json({ error: "Error al crear paciente" });
  }
};

// Actualizar paciente
exports.updatePaciente = async (req, res) => {
  try {
    const paciente = await Paciente.findByPk(req.params.id);
    if (!paciente) {
      logger.error("Paciente no encontrado por su ID");
      return res.status(404).json({ error: "Paciente no encontrado" });
    }
    await paciente.update(req.body);
    logger.info("Pacientes actualizado correctamente");
    res.json(paciente);
  } catch (error) {
    logger.error("Error al actualizar un Paciente: " + error.message);
    res.status(500).json({ error: "Error al actualizar paciente" });
  }
};

// Eliminar paciente
exports.deletePaciente = async (req, res) => {
  try {
    const paciente = await Paciente.findByPk(req.params.id);
    if (!paciente) {
      logger.error("Paciente no encontrado por su ID");
      return res.status(404).json({ error: "Paciente no encontrado" });
    }
    await paciente.destroy();
    logger.info("Pacientes eliminado correctamente");
    res.json({ message: "Paciente eliminado" });
  } catch (error) {
    logger.error("Error al elimininar un Paciente: " + error.message);
    res.status(500).json({ error: "Error al eliminar paciente" });
  }
};

// 🔍 Buscar Paciente por correo
exports.buscarPorCorreo = async (req, res) => {
  try {
    const { correo } = req.params;
    if (!correo) {
      logger.info("Debe proporcionar un correo para buscar un Paciente");
      return res.status(400).json({ error: "Debe proporcionar un correo" });
    }
    const paciente = await Paciente.findOne({ where: { Correo: correo } });
    if (!paciente) {
      logger.info("Paciente no encontrado por su correo");
      return res.json(null);
    }
    res.json(paciente);
    logger.info("Paciente encontrado por su correo");
  } catch (error) {
    logger.error("Error al buscar Paciente por correo: " + error.message);
    res.status(500).json({ error: "Error al buscar Paciente por correo" });
  }
};

// 🔍 Buscar Paciente por celular
exports.buscarPorCelular = async (req, res) => {
  try {
    const { celular } = req.params;
    if (!celular) {
      logger.info("Debe proporcionar un número de celular para buscar un Paciente");
      return res.status(400).json({ error: "Debe proporcionar un número de celular" });
    }
    const paciente = await Paciente.findOne({ where: { Celular: celular } });
    if (!paciente) {
      logger.info("Paciente no encontrado por su celular");
      return res.json(null);
    }
    logger.info("Paciente encontrado por su celular");
    res.json(paciente);
  } catch (error) {
    logger.error("Error al buscar Paciente por celular: " + error.message);
    res.status(500).json({ error: "Error al buscar Paciente por celular" });
  }
};

// 🔍 Buscar Paciente por documento
exports.buscarPorDocumento = async (req, res) => {
  try {
    const { documento } = req.params;
    if (!documento) {
      logger.info("Debe proporcionar un documento para buscar un Paciente");
      return res.status(400).json({ error: "Debe proporcionar un documento" });
    }
    const paciente = await Paciente.findOne({ where: { Documento: documento } });
    if (!paciente) {
      logger.info("Paciente no encontrado por su documento");
      return res.json(null);
    }
    logger.info("Paciente encontrado por su documento");
    res.json(paciente);
  } catch (error) {
    logger.error("Error al buscar Paciente por documento: " + error.message);
    res.status(500).json({ error: "Error al buscar Paciente por documento" });
  }
};

exports.buscarPorNroHistoriaClinica = async (req, res) => {
  try {
    const { historiaclinica } = req.params;
    if (!historiaclinica) {
      logger.info("Debe proporcionar un documento para buscar un Paciente");
      return res.status(400).json({ error: "Debe proporcionar un documento" });
    }
    const paciente = await Paciente.findOne({ where: { NroHistoriaClinica: historiaclinica } });
    if (!paciente) {
      logger.info("Paciente no encontrado por su nro de historia clinica");
      return res.json(null);
    }
    logger.info("Paciente encontrado por su documento");
    res.json(paciente);
  } catch (error) {
    logger.error("Error al buscar Paciente por documento: " + error.message);
    res.status(500).json({ error: "Error al buscar Paciente por documento" });
  }
};

// 🔍 Filtrar Pacientes por Apellidos y Nombres
exports.filtrarPorApellidosNombres = async (req, res) => {
  try {
    const { dato } = req.params;
    if (!dato) {
      logger.info("Debe proporcionar apellidos y nombres para filtrar Pacientes");
      return res.status(400).json({ error: "Debe proporcionar apellidos y nombres" });
    }
    const pacientes = await Paciente.findAll({
      where: {
        [Op.or]: [
          // Concatenar Apellidos + ' ' + Nombres
          Sequelize.where(
            Sequelize.fn('CONCAT', 
              Sequelize.col('Apellidos'), 
              ' ', 
              Sequelize.col('Nombres')
            ), 
            { [Op.like]: `%${dato}%` }
          ),
          // Concatenar Nombres + ' ' + Apellidos (por si viene en orden inverso)
          Sequelize.where(
            Sequelize.fn('CONCAT', 
              Sequelize.col('Nombres'), 
              ' ', 
              Sequelize.col('Apellidos')
            ), 
            { [Op.like]: `%${dato}%` }
          )
        ]
      }
    });
    logger.info("Pacientes filtrados por apellidos y nombres correctamente");
    res.json(pacientes);
  } catch (error) {
    logger.error("Error al filtrar Pacientes por Apellidos y Nombres: " + error.message);
    res.status(500).json({ error: "Error al filtrar Pacientes por Apellidos y Nombres" });
  }
}

// 🔍 Filtrar Pacientes por tipoDocumentoId
exports.filtrarPorTipoDocumento = async (req, res) => {
  try {
    const { tipoDocumento } = req.params;
    if (!tipoDocumento) {
      logger.info("Debe proporcionar un tipo de documento para filtrar Pacientes");
      return res.status(400).json({ error: "Debe proporcionar un tipo de documento" });
    }
    const pacientes = await Paciente.findAll({ where: { TipoDocumentoId: tipoDocumento } });
    logger.info("Pacientes filtrados por tipo de documento correctamente");
    res.json(pacientes);
  } catch (error) {
    logger.error("Error al filtrar Pacientes por tipoDocumentoId: " + error.message);
    res.status(500).json({ error: "Error al filtrar Pacientes por tipoDocumentoId" });
  }
};

// 🔍 Filtrar Pacientes por generoId
exports.filtrarPorGenero = async (req, res) => {
  try {
    const { genero } = req.params;
    if (!genero) {
      logger.info("Debe proporcionar un género para filtrar Pacientes");
      return res.status(400).json({ error: "Debe proporcionar un género" });
    }
    const pacientes = await Paciente.findAll({ where: { GeneroId: genero } });
    logger.info("Pacientes filtrados por género correctamente");
    res.json(pacientes);
  } catch (error) {
    logger.error("Error al filtrar Pacientes por generoId: " + error.message);
    res.status(500).json({ error: "Error al filtrar Pacientes por generoId" });
  }
};

// 🔍 Filtrar Pacientes por sedeId
exports.filtrarPorSede = async (req, res) => {
  try {
    const { sede } = req.params;
    if (!sede) {
      logger.info("Debe proporcionar una sede para filtrar Pacientes");
      return res.status(400).json({ error: "Debe proporcionar una sede" });
    }
    const pacientes = await Paciente.findAll({ where: { SedeId: sede } });
    logger.info("Pacientes filtrados por sede correctamente");
    res.json(pacientes);
  } catch (error) {
    logger.error("Error al filtrar Pacientes por sedeId: " + error.message);
    res.status(500).json({ error: "Error al filtrar Pacientes por sedeId" });
  }
};
// 🔍 Filtrar Pacientes por fuenteCaptacionId
exports.filtrarPorFuenteCaptacion = async (req, res) => {
  try {
    const { fuenteCaptacion } = req.params;
    if (!fuenteCaptacion) {
      logger.info("Debe proporcionar una sede para filtrar Pacientes");
      return res.status(400).json({ error: "Debe proporcionar una sede" });
    }
    const pacientes = await Paciente.findAll({ where: { FuenteCaptacionId: fuenteCaptacion } });
    logger.info("Pacientes filtrados por sede correctamente");
    res.json(pacientes);
  } catch (error) {
    logger.error("Error al filtrar Pacientes por sedeId: " + error.message);
    res.status(500).json({ error: "Error al filtrar Pacientes por sedeId" });
  }
};
// 🔍 Filtrar Pacientes por IAFAId
exports.filtrarPorIAFA = async (req, res) => {
  try {
    const { IAFA } = req.params;
    if (!IAFA) {
      logger.info("Debe proporcionar una sede para filtrar Pacientes");
      return res.status(400).json({ error: "Debe proporcionar una sede" });
    }
    const pacientes = await Paciente.findAll({ where: { IAFAId: IAFA } });
    logger.info("Pacientes filtrados por sede correctamente");
    res.json(pacientes);
  } catch (error) {
    logger.error("Error al filtrar Pacientes por sedeId: " + error.message);
    res.status(500).json({ error: "Error al filtrar Pacientes por sedeId" });
  }
};
// 🔍 Filtrar Pacientes por tipoAfiliacion
exports.filtrarPorTipoAfiliacion = async (req, res) => {
  try {
    const { tipoAfiliacion } = req.params;
    if (!tipoAfiliacion) {
      logger.info("Debe proporcionar una sede para filtrar Pacientes");
      return res.status(400).json({ error: "Debe proporcionar una sede" });
    }
    const pacientes = await Paciente.findAll({ where: { TipoAfiliacion: tipoAfiliacion } });
    logger.info("Pacientes filtrados por sede correctamente");
    res.json(pacientes);
  } catch (error) {
    logger.error("Error al filtrar Pacientes por sedeId: " + error.message);
    res.status(500).json({ error: "Error al filtrar Pacientes por sedeId" });
  }
};
// 🔍 Filtrar Pacientes por gradoInstruccionId
exports.filtrarPorGradoInstruccion = async (req, res) => {
  try {
    const { gradoInstruccion } = req.params;
    if (!gradoInstruccion) {
      logger.info("Debe proporcionar una sede para filtrar Pacientes");
      return res.status(400).json({ error: "Debe proporcionar una sede" });
    }
    const pacientes = await Paciente.findAll({ where: { GradoInstruccionId: gradoInstruccion } });
    logger.info("Pacientes filtrados por sede correctamente");
    res.json(pacientes);
  } catch (error) {
    logger.error("Error al filtrar Pacientes por sedeId: " + error.message);
    res.status(500).json({ error: "Error al filtrar Pacientes por sedeId" });
  }
};
// 🔍 Filtrar Pacientes por odontologoId
exports.filtrarPorOdontologo = async (req, res) => {
  try {
    const { odontologo } = req.params;
    if (!odontologo) {
      logger.info("Debe proporcionar una sede para filtrar Pacientes");
      return res.status(400).json({ error: "Debe proporcionar una sede" });
    }
    const pacientes = await Paciente.findAll({ where: { OdontologoId: odontologo } });
    logger.info("Pacientes filtrados por sede correctamente");
    res.json(pacientes);
  } catch (error) {
    logger.error("Error al filtrar Pacientes por sedeId: " + error.message);
    res.status(500).json({ error: "Error al filtrar Pacientes por sedeId" });
  }
};
