const {Pago, Perido} = require("../models");
const logger = require("../middlewares/logger.middleware");

// Obtener todas los Pagos
exports.getPagoAll = async (req, res) => {
  try {
    const pagos = await Pago.findAll();
    logger.info("Pagos obtenidos correctamente");
    res.json(pagos);
  } catch (error) {
    logger.error("Error al obtener Pagos: " + error.message);
    res.status(500).json({ error: "Error al obtener Pagoes" });
  }
};

// Obtener todas los Pagos por trabajador
exports.getPagoByTrabajador = async (req, res) => {
  try {
    const { trabajador } = req.params;
    const pagos = await Pago.findAll(
      {
        where: {
          PersonalId: trabajador
        },
        order: [['PeriodoId', 'DESC']] // Ordenar por fecharegistro en orden descendente
      }
    );
    logger.info("Pagos obtenidos correctamente");
    res.json(pagos);
  } catch (error) {
    logger.error("Error al obtener Pagos: " + error.message);
    res.status(500).json({ error: "Error al obtener Pagoes" });
  }
};

// Obtener todas los Pagos por trabajador
exports.getPagoByEgreso = async (req, res) => {
  try {
    const { egreso } = req.params;
    const pagos = await Pago.findAll(
      {
        where: {
          EgresoId: egreso
        },
        order: [['PeriodoId', 'DESC']] // Ordenar por fecharegistro en orden descendente
      }
    );
    logger.info("Pagos obtenidos correctamente");
    res.json(pagos);
  } catch (error) {
    logger.error("Error al obtener Pagos: " + error.message);
    res.status(500).json({ error: "Error al obtener Pagoes" });
  }
};

// Obtener una Pago por ID
exports.getPagoById = async (req, res) => {
  try {
    const pago = await Pago.findByPk(req.params.id);
    if (!pago) {
      logger.error("Pago no encontrada por su ID");
      return res.status(404).json({ error: "Pago no encontrada" });
    }
    logger.info("Pago obtenido correctamente por su ID");
    res.json(pago);
  } catch (error) {
    logger.error("Error al obtener el Pago mediante su Id: " + error.message);
    res.status(500).json({ error: "Error al obtener Pago" });
  }
};

// Crear una Pago
exports.createPago = async (req, res) => {
  try {
    const pago = await Pago.create(req.body);
    logger.info("Pagos creado correctamente");
    res.status(201).json(pago);
  } catch (error) {
    logger.error("Error al crear un Pago: " + error.message);
    res.status(500).json({ error: "Error al crear Pago" });
  }
};

// Actualizar una Pago
exports.updatePago = async (req, res) => {
  try {
    const pago = await Pago.findByPk(req.params.id);
    if (!pago) {
      logger.error("Pago no encontrada por su ID");
      return res.status(404).json({ error: "Pago no encontrada" });
    }
    await pago.update(req.body);
    logger.info("Pagos actualizado correctamente");
    res.json(pago);
  } catch (error) {
    logger.error("Error al actualizar un Pago: " + error.message);
    res.status(500).json({ error: "Error al actualizar Pago" });
  }
};

// Eliminar una Pago
exports.deletePago = async (req, res) => {
  try {
    const pago = await Pago.findByPk(req.params.id);
    if (!pago) {
      logger.error("Pago no encontrada por su ID");
      return res.status(404).json({ error: "Pago no encontrada" });
    }
    await pago.destroy();
    logger.info("Pagos eliminado correctamente");
    res.json({ message: "Pago eliminada" });
  } catch (error) {
    logger.error("Error al elimininar un Pago: " + error.message);
    res.status(500).json({ error: "Error al eliminar Pago" });
  }
};

