const {Proveedor, Sede} = require("../models");
const logger = require("../middlewares/logger.middleware");

// Obtener todas las Proveedores
exports.getProveedorAll = async (req, res) => {
  try {
    const proveedores = await Proveedor.findAll(
      {
        include: [
          {
            model: Sede,
            as: 'sede'
          }
        ]
      }
    );
    logger.info("Proveedor obtenidos correctamente");
    res.json(proveedores);
  } catch (error) {
    logger.error("Error al obtener Proveedor: " + error.message);
    res.status(500).json({ error: "Error al obtener Proveedores" });
  }
};
// Obtener una Proveedor por el hallazgo o incidencia
exports.getProveedorBySede = async (req, res) => {
  try {
    const { sede } = req.params;
    const proveedores = await Proveedor.findAll(
      {
        where: {
          SedeId: sede
        },
        include: [
          {
            model: Sede,
            as: 'sede'
          }
        ]
      }
    );
    if (!proveedores) {
      logger.error("Proveedor no encontrada por su ID");
      return res.status(404).json({ error: "Proveedor no encontrada" });
    }
    logger.info("Proveedor obtenido correctamente por su ID");
    res.json(Proveedores);
  } catch (error) {
    logger.error("Error al obtener el Proveedor mediante su Id: " + error.message);
    res.status(500).json({ error: "Error al obtener Proveedor" });
  }
};
// Obtener una Proveedor por ID
exports.getProveedorById = async (req, res) => {
  try {
    const proveedor = await Proveedor.findByPk(req.params.id);
    if (!proveedor) {
      logger.error("Proveedor no encontrada por su ID");
      return res.status(404).json({ error: "Proveedor no encontrada" });
    }
    logger.info("Proveedor obtenido correctamente por su ID");
    res.json(proveedor);
  } catch (error) {
    logger.error("Error al obtener el Proveedor mediante su Id: " + error.message);
    res.status(500).json({ error: "Error al obtener Proveedor" });
  }
};

// Crear una Proveedor
exports.createProveedor = async (req, res) => {
  try {
    const proveedor = await Proveedor.create(req.body);
    logger.info("Proveedor creado correctamente");
    res.status(201).json(proveedor);
  } catch (error) {
    logger.error("Error al crear un Proveedor: " + error.message);
    res.status(500).json({ error: "Error al crear Proveedor" });
  }
};

// Actualizar una Proveedor
exports.updateProveedor = async (req, res) => {
  try {
    const proveedor = await Proveedor.findByPk(req.params.id);
    if (!proveedor) {
      logger.error("Proveedor no encontrada por su ID");
      return res.status(404).json({ error: "Proveedor no encontrada" });
    }
    await proveedor.update(req.body);
    logger.info("Proveedor actualizado correctamente");
    res.json(Proveedor);
  } catch (error) {
    logger.error("Error al actualizar un Proveedor: " + error.message);
    res.status(500).json({ error: "Error al actualizar Proveedor" });
  }
};

// Eliminar una Proveedor
exports.deleteProveedor = async (req, res) => {
  try {
    const proveedor = await Proveedor.findByPk(req.params.id);
    if (!proveedor) {
      logger.error("Proveedor no encontrada por su ID");
      return res.status(404).json({ error: "Proveedor no encontrada" });
    }
    await Proveedor.destroy();
    logger.info("Proveedor eliminado correctamente");
    res.json({ message: "Proveedor eliminada" });
  } catch (error) {
    logger.error("Error al elimininar un Proveedor: " + error.message);
    res.status(500).json({ error: "Error al eliminar Proveedor" });
  }
};

