const {Receta} = require("../models");
const logger = require("../middlewares/logger.middleware");

// Obtener todas las Recetaes
exports.getRecetaAll = async (req, res) => {
  try {
    const datos = await Receta.findAll();
    logger.info("Recetas obtenidos correctamente");
    res.json(datos);
  } catch (error) {
    logger.error("Error al obtener Recetas: " + error.message);
    res.status(500).json({ error: "Error al obtener Recetaes" });
  }
};

// Obtener Receta por Historia Clinica
exports.getRecetaByPaciente = async (req, res) => {
  try {
    const { paciente } = req.params;
    const datos = await Receta.findAll(
      {
        where: {
          PacienteId: paciente
        }
      }
    );
    if (!datos) {
      logger.error("Receta no encontrada por su asociacion con Historia Clinica");
      return res.status(404).json({ error: "Recetas no encontrada" });
    }
    logger.info("Recetas obtenidas correctamente por su asociacion con la Historia Clinica");
    res.json(datos);
  } catch (error) {
    logger.error("Error al obtener el Receta mediante por Historia Clinica: " + error.message);
    res.status(500).json({ error: "Error al obtener Recetas" });
  }
};

// Obtener una Receta por ID
exports.getRecetaById = async (req, res) => {
  try {
    const dato = await Receta.findByPk(req.params.id);
    if (!dato) {
      logger.error("Receta no encontrada por su ID");
      return res.status(404).json({ error: "Receta no encontrada" });
    }
    logger.info("Receta obtenido correctamente por su ID");
    res.json(dato);
  } catch (error) {
    logger.error("Error al obtener el Receta mediante su Id: " + error.message);
    res.status(500).json({ error: "Error al obtener Receta" });
  }
};

// Crear una Receta
exports.createReceta = async (req, res) => {
  try {
    const dato = await Receta.create(req.body);
    logger.info("Recetas creado correctamente");
    res.status(201).json(dato);
  } catch (error) {
    logger.error("Error al crear un Receta: " + error.message);
    res.status(500).json({ error: "Error al crear Receta" });
  }
};

// Actualizar una Receta
exports.updateReceta = async (req, res) => {
  try {
    const dato = await Receta.findByPk(req.params.id);
    if (!dato) {
      logger.error("Receta no encontrada por su ID");
      return res.status(404).json({ error: "Receta no encontrada" });
    }
    await dato.update(req.body);
    logger.info("Recetas actualizado correctamente");
    res.json(dato);
  } catch (error) {
    logger.error("Error al actualizar un Receta: " + error.message);
    res.status(500).json({ error: "Error al actualizar Receta" });
  }
};

// Eliminar una Receta
exports.deleteReceta = async (req, res) => {
  try {
    const dato = await Receta.findByPk(req.params.id);
    if (!dato) {
      logger.error("Receta no encontrada por su ID");
      return res.status(404).json({ error: "Receta no encontrada" });
    }
    await Receta.destroy();
    logger.info("Recetas eliminado correctamente");
    res.json({ message: "Receta eliminada" });
  } catch (error) {
    logger.error("Error al elimininar un Receta: " + error.message);
    res.status(500).json({ error: "Error al eliminar Receta" });
  }
};

