const { Op, fn, col, Sequelize } = require('sequelize');
const {Usuario,Paciente,Ingreso,Egreso} = require("../models");

// Obtener usuarios por fecha de cumpleaños
exports.getUsersByBirthday = async(req, res, next) => {
  try {
    const { month, day } = req.body;

    // Validate input
    if (!month || !day) {
      return res.status(400).json({
        success: false,
        message: 'Mes y día son requeridos'
      });
    }

    // Ensure month and day are valid
    const monthNum = parseInt(month);
    const dayNum = parseInt(day);
    
    if (monthNum < 1 || monthNum > 12 || dayNum < 1 || dayNum > 31) {
      return res.status(400).json({
        success: false,
        message: 'Mes o día inválidos'
      });
    }

    // Query users with matching birthday (month and day)
    const users = await Usuario.findAll(
      {
        where: {
          [Op.and]: [
            Sequelize.where(
            Sequelize.fn('EXTRACT', Sequelize.literal('MONTH FROM "FechaNacimiento"')),
            monthNum
          ),
          Sequelize.where(
            Sequelize.fn('EXTRACT', Sequelize.literal('DAY FROM "FechaNacimiento"')),
            dayNum
          )
          ]
        },
        attributes: ['Nombre', 'Apellido', 'FechaNacimiento', 'Correo'] // Include relevant fields
      }
    );

    // Check if any users were found
    if (!users || users.length === 0) {
      return res.json({
        success: false,
        message: 'Usuario no encontrado con cumpleaños en esta fecha'
      });
    }

    // Format response
    const response = users.map(user => ({
      NombreCompleto: user.Nombre + ' ' + user.Apellido,
      Correo: user.Correo,
      FechaNacimiento: user.FechaNacimiento
    }));

    return res.status(200).json({
      success: true,
      message: 'Usuarios con cumpleaños coincidente obtenidos correctamente',
      data: response
    });

  } catch (error) {
    console.error('Error in getUsersByBirthday:', error);
    next(error);
  }
}


// Obtener pacientes por fecha de cumpleaños
exports.getPacientesByBirthday = async(req, res, next) => {
  try {
    const { month, day } = req.body;

    // Validate input
    if (!month || !day) {
      return res.status(400).json({
        success: false,
        message: 'Mes y día son requeridos'
      });
    }

    // Ensure month and day are valid
    const monthNum = parseInt(month);
    const dayNum = parseInt(day);
    
    if (monthNum < 1 || monthNum > 12 || dayNum < 1 || dayNum > 31) {
      return res.status(400).json({
        success: false,
        message: 'Mes y día inválidos'
      });
    }

    // Query users with matching birthday (month and day)
    const pacientes = await Paciente.findAll(
      {
        where: {
          [Op.and]: [
            Sequelize.where(
            Sequelize.fn('EXTRACT', Sequelize.literal('MONTH FROM "FechaNacimiento"')),
            monthNum
          ),
          Sequelize.where(
            Sequelize.fn('EXTRACT', Sequelize.literal('DAY FROM "FechaNacimiento"')),
            dayNum
          )
          ]
        },
        attributes: ['Nombres', 'Apellidos', 'FechaNacimiento', 'Correo'] // Include relevant fields
      }
    );

    // Check if any users were found
    if (!pacientes || pacientes.length === 0) {
      return res.json({
        success: false,
        message: 'No hay Pacientes que tengan cumpleaños hoy'
      });
    }

    // Format response
    const response = pacientes.map(paciente => ({
      NombreCompleto: paciente.Nombres + ' ' + paciente.Apellidos,
      Correo: paciente.Correo,
      FechaNacimiento: paciente.FechaNacimiento
    }));

    return res.status(200).json({
      success: true,
      message: 'Pacientes con cumpleaños coincidente obtenidos correctamente',
      data: response
    });

  } catch (error) {
    console.error('Error in getUsersByBirthday:', error);
    next(error);
  }
}

// Obtener total de ingresos por sede en un rango de fechas
exports.getEgresoTotalBySede = async(req, res, next) => {
  try {
    const { fechaInicio, fechaFin } = req.body;

    // Validate input
    if (!fechaInicio || !fechaFin) {
      return res.status(400).json({
        success: false,
        message: 'Start date (fechaInicio) and end date (fechaFin) are required'
      });
    }

    // Validate date format
    const startDate = new Date(fechaInicio);
    const endDate = new Date(fechaFin);
    if (isNaN(startDate) || isNaN(endDate)) {
      return res.status(400).json({
        success: false,
        message: 'Invalid date format'
      });
    }

    // Query total amount by Sede
    const results = await Egreso.findAll({
      where: {
        FechaRegistro: {
          [Op.between]: [startDate, endDate]
        }
      },
      attributes: [
        'SedeId',
        [fn('SUM', col('MontoSoles')), 'totalMonto']
      ],
      group: ['SedeId'],
      raw: true
    });

    // Check if any results were found
    if (!results || results.length === 0) {
      return res.json({
        success: false,
        message: 'No records found for the specified date range'
      });
    }

    return res.status(200).json({
      success: true,
      message: 'Total amount by location retrieved successfully',
      data: results
    });

  } catch (error) {
    console.error('Error in getTotalBySede:', error);
    next(error);
  }
}

// Obtener total de ingresos por sede en un rango de fechas
exports.getIngresoTotalBySede = async(req, res, next) => {
  try {
    const { fechaInicio, fechaFin } = req.body;

    // Validate input
    if (!fechaInicio || !fechaFin) {
      return res.status(400).json({
        success: false,
        message: 'Start date (fechaInicio) and end date (fechaFin) are required'
      });
    }

    // Validate date format
    const startDate = new Date(fechaInicio);
    const endDate = new Date(fechaFin);
    if (isNaN(startDate) || isNaN(endDate)) {
      return res.status(400).json({
        success: false,
        message: 'Invalid date format'
      });
    }

    // Query total amount by Sede
    const results = await Ingreso.findAll({
      where: {
        TipoProveedorId: 1, // Assuming TipoProveedorId 1 is for Odontologo
        FechaRegistro: {
          [Op.between]: [startDate, endDate]
        }
      },
      attributes: [
        'SedeId',
        [fn('SUM', col('MontoSoles')), 'totalMonto']
      ],
      group: ['SedeId'],
      raw: true
    });

    // Check if any results were found
    if (!results || results.length === 0) {
      return res.status(404).json({
        success: false,
        message: 'No records found for the specified date range'
      });
    }

    return res.status(200).json({
      success: true,
      message: 'Total amount by location retrieved successfully',
      data: results
    });

  } catch (error) {
    console.error('Error in getTotalBySede:', error);
    next(error);
  }
}

// Obtener total de ingresos por sede y odontologo en un rango de fechas
exports.getIngresoTotalBySedeAndOdontologo = async(req, res, next) => {
  try {
    const { fechaInicio, fechaFin } = req.body;

    // Validate input
    if (!fechaInicio || !fechaFin) {
      return res.status(400).json({
        success: false,
        message: 'Start date (fechaInicio) and end date (fechaFin) are required'
      });
    }

    // Validate date format
    const startDate = new Date(fechaInicio);
    const endDate = new Date(fechaFin);
    if (isNaN(startDate) || isNaN(endDate)) {
      return res.status(400).json({
        success: false,
        message: 'Invalid date format'
      });
    }

    // Query total amount by Sede and Odontologo
    const results = await Ingreso.findAll({
      where: {
        TipoProveedorId: 1, // Assuming TipoProveedorId 1 is for Odontologo
        FechaRegistro: {
          [Op.between]: [startDate, endDate]
        }
      },
      attributes: [
        'SedeId',
        'Odontologo',
        [fn('SUM', col('MontoSoles')), 'totalMonto']
      ],
      group: ['SedeId', 'Odontologo'],
      raw: true
    });

    // Check if any results were found
    if (!results || results.length === 0) {
      return res.json({
        success: false,
        message: 'No records found for the specified date range'
      });
    }

    return res.status(200).json({
      success: true,
      message: 'Total amount by location and dentist retrieved successfully',
      data: results
    });

  } catch (error) {
    console.error('Error in getTotalBySedeAndOdontologo:', error);
    next(error);
  }
}