const {Tarifario, Catalogo} = require("../models");
const logger = require("../middlewares/logger.middleware");

// Obtener todas las Tarifarioes
exports.getTarifarioAll = async (req, res) => {
  try {
    const tarifarios = await Tarifario.findAll(
      {
        include: [
          {
            model: Catalogo,
            as: "catalogo"
          }
        ],
        order: [["createdAt", "DESC"]],
      }
    );
    logger.info("Tarifarios obtenidos correctamente");
    res.json(tarifarios);
  } catch (error) {
    logger.error("Error al obtener Tarifarios: " + error.message);
    res.status(500).json({ error: "Error al obtener Tarifarioes" });
  }
};

// Filtrar Tarifarioes por IAFA
exports.filtrarbyIAFA = async (req, res) => {
  try {
    const { iafa } = req.params;
    const tarifarios = await Tarifario.findAll(
      {
        where: {
          IAFAId: iafa
        },
        include: [
          {
            model: Catalogo,
            as: "catalogo"
          }
        ],
      }
    );

    res.json(tarifarios);
  } catch (error) {
    res.status(500).json({ error: "Error al filtrar Tarifarioes por odontólogo" });
  }
};

// Filtrar Tarifarioes por Catalogo
exports.filtrarbyCatalogo = async (req, res) => {
  try {
    const { catalogo } = req.params;
    const tarifarios = await Tarifario.findAll(
      {
        where: {
          CatalogoId: catalogo
        },
        include: [
          {
            model: Catalogo,
            as: "catalogo"
          }
        ],
      }
    );

    res.json(tarifarios);
  } catch (error) {
    res.status(500).json({ error: "Error al filtrar Tarifarioes por odontólogo" });
  }
};

// Obtener una Tarifario por ID
exports.getTarifarioById = async (req, res) => {
  try {
    const tarifario = await Tarifario.findByPk(req.params.id);
    if (!tarifario) {
      logger.error("Tarifario no encontrada por su ID");
      return res.status(404).json({ error: "Tarifario no encontrada" });
    }
    logger.info("Tarifario obtenido correctamente por su ID");
    res.json(tarifario);
  } catch (error) {
    logger.error("Error al obtener el Tarifario mediante su Id: " + error.message);
    res.status(500).json({ error: "Error al obtener Tarifario" });
  }
};

// Crear una Tarifario
exports.createTarifario = async (req, res) => {
  try {
    const tarifario = await Tarifario.create(req.body);
    logger.info("Tarifarios creado correctamente");
    res.status(201).json(tarifario);
  } catch (error) {
    logger.error("Error al crear un Tarifario: " + error.message);
    res.status(500).json({ error: "Error al crear Tarifario" });
  }
};

// Actualizar una Tarifario
exports.updateTarifario = async (req, res) => {
  try {
    const tarifario = await Tarifario.findByPk(req.params.id);
    if (!tarifario) {
      logger.error("Tarifario no encontrada por su ID");
      return res.status(404).json({ error: "Tarifario no encontrada" });
    }
    await tarifario.update(req.body);
    logger.info("Tarifarios actualizado correctamente");
    res.json(tarifario);
  } catch (error) {
    logger.error("Error al actualizar un Tarifario: " + error.message);
    res.status(500).json({ error: "Error al actualizar Tarifario" });
  }
};

// Eliminar una Tarifario
exports.deleteTarifario = async (req, res) => {
  try {
    const tarifario = await Tarifario.findByPk(req.params.id);
    if (!tarifario) {
      logger.error("Tarifario no encontrada por su ID");
      return res.status(404).json({ error: "Tarifario no encontrada" });
    }
    await Tarifario.destroy();
    logger.info("Tarifarios eliminado correctamente");
    res.json({ message: "Tarifario eliminada" });
  } catch (error) {
    logger.error("Error al elimininar un Tarifario: " + error.message);
    res.status(500).json({ error: "Error al eliminar Tarifario" });
  }
};
