const express = require("express");
const AccionController = require("../controllers/accion.controller");
const { verifyToken } = require('../middlewares/auth.middleware');

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Accion:
 *       type: object
 *       required:
 *         - Descripcion
 *         - ResponsableDescripcion
 *         - FechaInicio
 *         - InformadoPor
 *         - AnalisisdeCausas
 *         - ResponsableAnalisis
 *         - FechaAnalisis
 *         - Accion
 *         - TipoAccion
 *         - ResponsableAccion
 *         - FechaInicioAccion
 *         - FechaCierreAccion
 *         - Verificacion
 *         - ResponsableVerificacion
 *         - FechaFinalizacion
 *         - HallazgoId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único de la accion
 *         Descripcion:
 *           type: string
 *           description: Nombre de la Accion
 *         ResponsableDescripcion:
 *           type: string
 *           description: Resposable a proporcionar la informacion
 *         FechaInicio:
 *           type: Date
 *           description: Inicio de la accion toma en cuenta
 *         InformadoPor:
 *           type: string
 *           description: Recepciono el Reclamo
 *         AnalisisdeCausas:
 *           type: string
 *           description: Analizas las posibles causas
 *         ResponsableAnalisis:
 *           type: string
 *           description: Persona que analizara las causas
 *         FechaAnalisis:
 *           type: Date
 *           description: Fecha en la que la persona lo analiza
 *         Accion:
 *           type: string
 *           description: Accion tomada para solucionar o prevenir
 *         TipoAccion:
 *           type: number
 *           description: Tipo de accion tomada (proceso administrativo, amonestacion, etc)
 *         ResponsableAccion:
 *           type: string
 *           description: Resposable a tomar accion
 *         FechaInicioAccion:
 *           type: Date
 *           description: Fecha de Inicio de la accion aplicada
 *         FechaCierreAccion:
 *           type: Date
 *           description: Fecha de cierre de la aplicacion de la accion
 *         Verificacion:
 *           type: number
 *           description: Verificacion de terceros (camaras, tetigos, etc)
 *         ResponsableVerificacion:
 *           type: string
 *           description: Quien se encargara de la verificacion
 *         FechaFinalizacion:
 *           type: Date
 *           description: Finalizacion de la incidencia
 *         HallazgoId:
 *           type: id
 *           description: A que incidencia va la accion
 *       example:
 *         id: 1
 *         Descripcion: "Descripcion de la Accion"
 *         ResponsableDescripcion: "Responsable de la Descripcion"
 *         FechaInicio: "2025-04-25"
 *         InformadoPor: "Informado por"
 *         AnalisisdeCausas: "Analisis de las causas"
 *         ResponsableAnalisis: "Responsable del Analisis"
 *         FechaAnalisis: "2025-04-25"
 *         Accion: "Accion que se tomo"
 *         TipoAccion: "Tipo de Accion Tomada"
 *         ResponsableAccion: "Responsable de la Accion"
 *         FechaInicioAccion: "2025-04-25"
 *         FechaCierreAccion: "2025-04-25"
 *         Verificacion: "Verificacion"
 *         ResponsableVerificacion: "Responsable de la Verificacion"
 *         FechaFinalizacion: "2025-04-25"
 *         HallazgoId: 1

 */
/**
 * @swagger
 * tags:
 *   name: Accion
 *   description: API para gestionar Accions
 */
/**
 * @swagger
 * /incidencias/accion:
 *   get:
 *     summary: Obtiene la lista de Accions
 *     tags: [Accion]
 *     responses:
 *       200:
 *         description: Lista de Accions.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Accion'
 */
router.get("/", verifyToken, AccionController.getAccionAll);
/**
 * @swagger
 * /incidencias/accion/incidencia/{incidencia}:
 *   get:
 *     summary: Obtiene Accions asociadas a una Incidencia
 *     tags: [Accion]
 *     parameters:
 *       - in: path
 *         name: incidencia
 *         schema:
 *           type: number
 *         required: true
 *         description: ID de la Incidencia
 *     responses:
 *       200:
 *         description: Accions encontradas de la Incidencia asociada.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Accion'
 *       404:
 *         description: Accions no encontrado por la Incidencia Escogida.
 */
router.get("/incidencia/:incidencia", verifyToken, AccionController.getAccionByIncidencia);
/**
 * @swagger
 * /incidencias/accion/{id}:
 *   get:
 *     summary: Obtiene un Accion por ID
 *     tags: [Accion]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Accion
 *     responses:
 *       200:
 *         description: Accion encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Accion'
 *       404:
 *         description: Accion no encontrado.
 */
router.get("/:id", verifyToken, AccionController.getAccionById);
/**
 * @swagger
 * /incidencias/accion/create:
 *   post:
 *     summary: Crea un nuevo Accion
 *     tags: [Accion]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Accion'
 *     responses:
 *       201:
 *         description: Accion creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Accion'
 */
router.post("/create", verifyToken, AccionController.createAccion);
/**
 * @swagger
 * /incidencias/accion/update/{id}:
 *   put:
 *     summary: Actualiza un Accion existente
 *     tags: [Accion]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Accion a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Accion'
 *     responses:
 *       200:
 *         description: Accion actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Accion'
 *       404:
 *         description: Accion no encontrado.
 */
router.put("/update/:id", verifyToken, AccionController.updateAccion);
/**
 * @swagger
 * /incidencias/accion/{id}:
 *   delete:
 *     summary: Elimina un Accion por ID
 *     tags: [Accion]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Accion a eliminar
 *     responses:
 *       200:
 *         description: Accion eliminado exitosamente.
 *       404:
 *         description: Accion no encontrado.
 */
router.delete("/:id", verifyToken, AccionController.deleteAccion);

module.exports = router;
