const express = require("express");
const AgendamientoController = require("../controllers/agendamiento.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware
// const { validatePerson } = require("../middlewares/validation.middleware");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Agendamiento:
 *       type: object
 *       required:
 *         - FechayHoraInicio
 *         - FechayHoraFin
 *         - Detalles
 *         - f
 *         - SedeNombre
 *         - SedeIdentificador
 *         - Servicio
 *         - UsuarioIdentificador
 *         - EstadoCitaId
 *         - ColorEstado
 *         - EstadoPagoId
 *         - TipoPagoId
 *         - PacienteId
 *         - UsuarioId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Agendamiento
 *         FechayHoraInicio:
 *           type: Date
 *           description: Nombre del Agendamiento
 *         FechayHoraFin:
 *           type: Date
 *           description: Descripcion del Agendamiento
 *         Detalles:
 *           type: string
 *           description: Foto del Agendamiento
 *         Referidopor:
 *           type: string
 *           description: Nombre del Agendamiento
 *         SedeNombre:
 *           type: string
 *           description: Descripcion del Agendamiento
 *         SedeIdentificador:
 *           type: number
 *           description: Foto del Agendamiento
 *         Servicio:
 *           type: string
 *           description: Nombre del Agendamiento
 *         UsuarioIdentificador:
 *           type: number
 *           description: Descripcion del Agendamiento
 *         EstadoCitaId:
 *           type: number
 *           description: Foto del Agendamiento
 *         ColorEstado:
 *           type: string
 *           description: Nombre del Agendamiento
 *         EstadoPagoId:
 *           type: number
 *           description: Descripcion del Agendamiento
 *         TipoPagoId:
 *           type: number
 *           description: Foto del Agendamiento
 *         PacienteId:
 *           type: number
 *           description: Nombre del Agendamiento
 *         UsuarioId:
 *           type: number
 *           description: Descripcion del Agendamiento
 *       example:
 *         id: "1"
 *         FechayHoraInicio: "Odontopediatría"
 *         FechayHoraFin: "Odontopediatría"
 *         Detalles: "Odontopediatría"
 *         Referidopor: "Odontopediatría"
 *         SedeNombre: "Odontopediatría"
 *         SedeIdentificador: "Odontopediatría"
 *         Servicio: "Odontopediatría"
 *         UsuarioIdentificador: "Odontopediatría"
 *         EstadoCitaId: "Odontopediatría"
 *         ColorEstado: "Odontopediatría"
 *         EstadoPagoId: "Odontopediatría"
 *         TipoPagoId: "Odontopediatría"
 *         PacienteId: "Odontopediatría"
 *         UsuarioId: "Odontopediatría"
 */
/**
 * @swagger
 * tags:
 *   name: Agendamiento
 *   description: API para gestionar Agendamiento
 */
/**
 * @swagger
 * /agenda/agendamiento:
 *   get:
 *     summary: Obtiene la lista de Agendamiento
 *     tags: [Agendamiento]
 *     responses:
 *       200:
 *         description: Lista de Agendamiento.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Agendamiento'
 */
router.get("/", verifyToken, AgendamientoController.getAgendamientoAll);
/**
 * @swagger
 * /agenda/agendamiento/{id}:
 *   get:
 *     summary: Obtiene un Agendamiento por ID
 *     tags: [Agendamiento]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Agendamiento
 *     responses:
 *       200:
 *         description: Agendamiento encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Agendamiento'
 *       404:
 *         description: Agendamiento no encontrado.
 */
router.get("/:id", verifyToken, AgendamientoController.getAgendamientoById);
/**
 * @swagger
 * /agenda/agendamiento/create:
 *   post:
 *     summary: Crea un nuevo Agendamiento
 *     tags: [Agendamiento]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Agendamiento'
 *     responses:
 *       201:
 *         description: Agendamiento creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Agendamiento'
 */
router.post("/create", verifyToken, AgendamientoController.createAgendamiento); // Aplicamos validación
/**
 * @swagger
 * /agenda/agendamiento/update/{id}:
 *   put:
 *     summary: Actualiza un Agendamiento existente
 *     tags: [Agendamiento]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Agendamiento a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Agendamiento'
 *     responses:
 *       200:
 *         description: Agendamiento actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Agendamiento'
 *       404:
 *         description: Agendamiento no encontrado.
 */
router.put("/update/:id", verifyToken, AgendamientoController.updateAgendamiento);
/**
 * @swagger
 * /agenda/agendamiento/{id}:
 *   delete:
 *     summary: Elimina un Agendamiento por ID
 *     tags: [Agendamiento]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Agendamiento a eliminar
 *     responses:
 *       200:
 *         description: Agendamiento eliminado exitosamente.
 *       404:
 *         description: Agendamiento no encontrado.
 */
router.delete("/:id", verifyToken, AgendamientoController.deleteAgendamiento);
// 🔍 Rutas individuales para filtrado
/**
 * @swagger
 * /agenda/agendamiento/filtrar/sedeypersonal/{sede}/{personal}:
 *   get:
 *     summary: Obtiene un Agendamiento por sede y personal
 *     tags: [Agendamiento]
 *     parameters:
 *       - in: path
 *         name: sede
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del sede
 *       - in: path
 *         name: personal
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del personal
 *     responses:
 *       200:
 *         description: Agendamiento encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Agendamiento'
 *       404:
 *         description: Agendamiento no encontrado.
 */
// ruta para filtrar citas por sede y odontologo con todos los estados
router.get("/filtrar/sedeypersonal/:sede/:personal", verifyToken, AgendamientoController.getAgendamientoBySedeandPersonal);
/**
 * @swagger
 * /agenda/agendamiento/filtrar/personal/{personal}:
 *   get:
 *     summary: Obtiene un Agendamiento por personal
 *     tags: [Agendamiento]
 *     parameters:
 *       - in: path
 *         name: personal
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del personal (odontologo)
 *     responses:
 *       200:
 *         description: Agendamiento encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Agendamiento'
 *       404:
 *         description: Agendamiento no encontrado.
 */
// ruta para filtrar las citas del paciente
router.get("/filtrar/personal/:personal", verifyToken, AgendamientoController.getAgendamientoByPersonal);
/**
 * @swagger
 * /agenda/agendamiento/filtrar/paciente/{paciente}:
 *   get:
 *     summary: Obtiene un Agendamiento por paciente
 *     tags: [Agendamiento]
 *     parameters:
 *       - in: path
 *         name: paciente
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del paciente
 *     responses:
 *       200:
 *         description: Agendamiento encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Agendamiento'
 *       404:
 *         description: Agendamiento no encontrado.
 */
// ruta para filtrar las citas del paciente
router.get("/filtrar/paciente/:paciente", verifyToken, AgendamientoController.getAgendamientoByPaciente);

module.exports = router;