const express = require("express");
const almacenController = require("../controllers/almacen.controller");
const { verifyToken } = require('../middlewares/auth.middleware');

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Almacen:
 *       type: object
 *       required:
 *         - Nombre
 *         - Ubicacion
 *         - SedeId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único de la Almacen
 *         Nombre:
 *           type: string
 *           description: Nombre del Almacen
 *         Ubicacion:
 *           type: string
 *           description: Ubicacion del Almacen
 *         SedeId:
 *           type: number
 *           description: Almacen pertenece a la Sede
 *       example:
 *         id: "1"
 *         Nombre: "Almacen Sede Principal"
 *         Ubicacion: "Ubicacion del Almacen en la Sede Principal"
 *         SedeId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Almacen
 *   description: API para gestionar Almacens
 */
/**
 * @swagger
 * /inventario/almacen:
 *   get:
 *     summary: Obtiene la lista de Almacens
 *     tags: [Almacen]
 *     responses:
 *       200:
 *         description: Lista de Almacens.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Almacen'
 */
router.get("/", verifyToken, almacenController.getAlmacenAll);
/**
 * @swagger
 * /inventario/almacen/sede/{sede}:
 *   get:
 *     summary: Obtiene un Almacen por ID
 *     tags: [Almacen]
 *     parameters:
 *       - in: path
 *         name: sede
 *         schema:
 *           type: number
 *         required: true
 *         description: Sede a la que pertenece el almacen
 *     responses:
 *       200:
 *         description: Almacenes encontrados.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Almacen'
 *       404:
 *         description: Almacen no encontrado.
 */
router.get("/sede/:sede", verifyToken, almacenController.getAlmacenBySede);
/**
 * @swagger
 * /inventario/almacen/responsable/{responsable}:
 *   get:
 *     summary: Obtiene un Almacen por responsable
 *     tags: [Almacen]
 *     parameters:
 *       - in: path
 *         name: responsable
 *         schema:
 *           type: number
 *         required: true
 *         description: responsable del almacen
 *     responses:
 *       200:
 *         description: Almacenes encontrados.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Almacen'
 *       404:
 *         description: Almacen no encontrado.
 */
router.get("/responsable/:responsable", verifyToken, almacenController.getAlmacenByResponsable);
/**
 * @swagger
 * /inventario/almacen/{id}:
 *   get:
 *     summary: Obtiene un Almacen por ID
 *     tags: [Almacen]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Almacen
 *     responses:
 *       200:
 *         description: Almacen encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Almacen'
 *       404:
 *         description: Almacen no encontrado.
 */
router.get("/:id", verifyToken, almacenController.getAlmacenById);
/**
 * @swagger
 * /inventario/almacen/create:
 *   post:
 *     summary: Crea un nuevo Almacen
 *     tags: [Almacen]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Almacen'
 *     responses:
 *       201:
 *         description: Almacen creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Almacen'
 */
router.post("/create", verifyToken, almacenController.createAlmacen);
/**
 * @swagger
 * /inventario/almacen/update/{id}:
 *   put:
 *     summary: Actualiza un Almacen existente
 *     tags: [Almacen]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Almacen a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Almacen'
 *     responses:
 *       200:
 *         description: Almacen actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Almacen'
 *       404:
 *         description: Almacen no encontrado.
 */
router.put("/update/:id", verifyToken, almacenController.updateAlmacen);
/**
 * @swagger
 * /inventario/almacen/{id}:
 *   delete:
 *     summary: Elimina un Almacen por ID
 *     tags: [Almacen]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Almacen a eliminar
 *     responses:
 *       200:
 *         description: Almacen eliminado exitosamente.
 *       404:
 *         description: Almacen no encontrado.
 */
router.delete("/:id", almacenController.deleteAlmacen);

// Rutas de filtrado individual


module.exports = router;
