const express = require("express");
const AtiendeController = require("../controllers/atiende.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Atiende:
 *       type: object
 *       required:
 *         - Nombre
 *         - ConsultorioId
 *         - UsuarioId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Atiende
 *         Nombre:
 *           type: string
 *           description: Nombre del Atiende
 *         ConsultorioId:
 *           type: number
 *           description: Indentificador del Conultorio
 *         UsuarioId:
 *           type: number
 *           description: Indentificador del Usuario
 *       example:
 *         id: "1"
 *         Nombre: "Odontopediatría"
 *         ConsultorioId: "1"
 *         UsuarioId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Atiende
 *   description: API para gestionar Atiende
 */
/**
 * @swagger
 * /agenda/atiende:
 *   get:
 *     summary: Obtiene la lista de Atiende
 *     tags: [Atiende]
 *     responses:
 *       200:
 *         description: Lista de Atiende.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Atiende'
 */
router.get("/", verifyToken, AtiendeController.getAtiendeAll);
/**
 * @swagger
 * /agenda/atiende/{id}:
 *   get:
 *     summary: Obtiene un Atiende por ID
 *     tags: [Atiende]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Atiende
 *     responses:
 *       200:
 *         description: Atiende encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Atiende'
 *       404:
 *         description: Atiende no encontrado.
 */
router.get("/:id", verifyToken, AtiendeController.getAtiendeById);
/**
 * @swagger
 * /agenda/atiende/create:
 *   post:
 *     summary: Crea un nuevo Atiende
 *     tags: [Atiende]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Atiende'
 *     responses:
 *       201:
 *         description: Atiende creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Atiende'
 */
router.post("/create", verifyToken, AtiendeController.createAtiende);
/**
 * @swagger
 * /agenda/atiende/update/{id}:
 *   put:
 *     summary: Actualiza un Atiende existente
 *     tags: [Atiende]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Atiende a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Atiende'
 *     responses:
 *       200:
 *         description: Atiende actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Atiende'
 *       404:
 *         description: Atiende no encontrado.
 */
router.put("/update/:id", verifyToken, AtiendeController.updateAtiende);
/**
 * @swagger
 * /agenda/atiende/{id}:
 *   delete:
 *     summary: Elimina un Atiende por ID
 *     tags: [Atiende]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Atiende a eliminar
 *     responses:
 *       200:
 *         description: Atiende eliminado exitosamente.
 *       404:
 *         description: Atiende no encontrado.
 */
router.delete("/:id", verifyToken, AtiendeController.deleteAtiende);
/**
 * @swagger
 * /agenda/atiende/filtrar/usuario/{usuario}:
 *   get:
 *     summary: Obtiene un Atiende por usuario
 *     tags: [Atiende]
 *     parameters:
 *       - in: path
 *         name: usuario
 *         schema:
 *           type: number
 *         required: true
 *         description: usuario del Atiende
 *     responses:
 *       200:
 *         description: Atiende encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Atiende'
 *       404:
 *         description: Atiende no encontrado.
 */
// 🔍 Rutas individuales para búsqueda
router.get("/filtrar/usuario/:usuario", verifyToken, AtiendeController.getAtiendeByOdontologo);
/**
 * @swagger
 * /agenda/atiende/filtrar/consultorio/{consultorio}:
 *   get:
 *     summary: Obtiene un Atiende por consultorio
 *     tags: [Atiende]
 *     parameters:
 *       - in: path
 *         name: consultorio
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Atiende
 *     responses:
 *       200:
 *         description: Atiende encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Atiende'
 *       404:
 *         description: Atiende no encontrado.
 */
router.get("/filtrar/consultorio/:consultorio", verifyToken, AtiendeController.getAtiendeByConsultorio);

module.exports = router;