const express = require("express");
const CargoUsuarioController = require("../controllers/cargousuario.controller");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     CargoUsuario:
 *       type: object
 *       required:
 *         - Nombre
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del usuario
 *         Nombre:
 *           type: string
 *           description: Cargo de usuario que usara el sistema
 *       example:
 *         id: "1"
 *         Nombre: "Administrador"
 */
/**
 * @swagger
 * tags:
 *   name: CargoUsuario
 *   description: API para gestionar Cargos de usuario
 */
/**
 * @swagger
 * /pilares/cargousuario:
 *   get:
 *     summary: Obtiene la lista de Cargos de Usuario
 *     tags: [CargoUsuario]
 *     responses:
 *       200:
 *         description: Lista de Cargos de Usuario.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/CargoUsuario'
 */
router.get(
    "/",
    CargoUsuarioController.getCargoUsuarioAll
);
/**
 * @swagger
 * /pilares/cargousuario/{id}:
 *   get:
 *     summary: Obtiene un Cargo de usuario por ID
 *     tags: [CargoUsuario]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Cargo de usuario
 *     responses:
 *       200:
 *         description: Cargo Usuario encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/CargoUsuario'
 *       404:
 *         description: Cargo Usuario no encontrado.
 */
router.get(
    "/:id",
    CargoUsuarioController.getCargoUsuarioById
);
/**
 * @swagger
 * /pilares/cargousuario/create:
 *   post:
 *     summary: Crea un nuevo Cargo Usuario
 *     tags: [CargoUsuario]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/CargoUsuario'
 *     responses:
 *       201:
 *         description: Cargo Usuario creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/CargoUsuario'
 */
router.post(
    "/create",
    CargoUsuarioController.createCargoUsuario
);
/**
 * @swagger
 * /pilares/cargousuario/update/{id}:
 *   put:
 *     summary: Actualiza un Cargo Usuario existente
 *     tags: [CargoUsuario]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Cargo Usuario a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/CargoUsuario'
 *     responses:
 *       200:
 *         description: Cargo Usuario actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/CargoUsuario'
 *       404:
 *         description: Cargo Usuario no encontrado.
 */
router.put(
    "/update/:id", 
    CargoUsuarioController.updateCargoUsuario
);
/**
 * @swagger
 * /pilares/cargousuario/{id}:
 *   delete:
 *     summary: Elimina un Cargo Usuario por ID
 *     tags: [CargoUsuario]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Cargo Usuario a eliminar
 *     responses:
 *       200:
 *         description: Cargo Usuario eliminado exitosamente.
 *       404:
 *         description: Cargo Usuario no encontrado.
 */
router.delete(
    "/:id",
    CargoUsuarioController.deleteCargoUsuario
);

module.exports = router;
