const express = require("express");
const CatalogoController = require("../controllers/catalogo.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Catalogo:
 *       type: object
 *       required:
 *         - Nombre
 *         - Descripcion
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Catalogo
 *         Nombre:
 *           type: string
 *           description: Nombre del Catalogo
 *         Descripcion:
 *           type: string
 *           description: Descripcion
 *       example:
 *         id: "1"
 *         Nombre: "Curacion de Carie Simple"
 *         Descripcion: "Se procede a la curacion de una carie simple"
 */
/**
 * @swagger
 * tags:
 *   name: Catalogo
 *   description: API para gestionar Catalogo
 */
/**
 * @swagger
 * /economia/catalogo:
 *   get:
 *     summary: Obtiene la lista de Catalogo
 *     tags: [Catalogo]
 *     responses:
 *       200:
 *         description: Lista de Catalogo.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Catalogo'
 */
router.get("/", verifyToken, CatalogoController.getCatalogoAll);
/**
 * @swagger
 * /economia/catalogo/{id}:
 *   get:
 *     summary: Obtiene un Catalogo por ID
 *     tags: [Catalogo]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Catalogo
 *     responses:
 *       200:
 *         description: Catalogo encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Catalogo'
 *       404:
 *         description: Catalogo no encontrado.
 */
router.get("/:id", verifyToken, CatalogoController.getCatalogoById);
/**
 * @swagger
 * /economia/catalogo/create:
 *   post:
 *     summary: Crea un nuevo Catalogo
 *     tags: [Catalogo]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Catalogo'
 *     responses:
 *       201:
 *         description: Catalogo creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Catalogo'
 */
router.post("/create", verifyToken, CatalogoController.createCatalogo);
/**
 * @swagger
 * /economia/catalogo/update/{id}:
 *   put:
 *     summary: Actualiza un Catalogo existente
 *     tags: [Catalogo]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Catalogo a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Catalogo'
 *     responses:
 *       200:
 *         description: Catalogo actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Catalogo'
 *       404:
 *         description: Catalogo no encontrado.
 */
router.put("/update/:id", verifyToken, CatalogoController.updateCatalogo);
/**
 * @swagger
 * /economia/catalogo/{id}:
 *   delete:
 *     summary: Elimina un Catalogo por ID
 *     tags: [Catalogo]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Catalogo a eliminar
 *     responses:
 *       200:
 *         description: Catalogo eliminado exitosamente.
 *       404:
 *         description: Catalogo no encontrado.
 */
router.delete("/:id", verifyToken, CatalogoController.deleteCatalogo);

module.exports = router;
