const express = require("express");
const ClienteController = require("../controllers/cliente.controller");
const { verifyToken } = require('../middlewares/auth.middleware');

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Cliente:
 *       type: object
 *       required:
 *         - Nombre
 *         - Apellido
 *         - TipoDocumentoId
 *         - NumeroDocumento
 *         - Celular
 *         - Correo
 *         - Apoderado
 *         - Direccion
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del cliente
 *         Nombre:
 *           type: string
 *           description: Nombre del Cliente
 *         Apellido:
 *           type: string
 *           description: Apellido del Cliente
 *         TipoDocumentoId:
 *           type: number
 *           description: Nombre del Cliente
 *         NumeroDocumento:
 *           type: string
 *           description: Numero de Documento del Cliente
 *         Celular:
 *           type: string
 *           description: Celular del Cliente
 *         Correo:
 *           type: string
 *           description: Correo del Cliente
 *         Apoderado:
 *           type: string
 *           description: Apoderado del Cliente; si es menor de edad
 *         Direccion:
 *           type: string
 *           description: Direccion del Cliente
 *       example:
 *         id: "1"
 *         Nombre: "Anibal Jhamil"
 *         Apellido: "Huaman Verastein"
 *         TipoDocumentoId: 1
 *         NumeroDocumento: "72547410"
 *         Celular: "+51958899251"
 *         Correo: "anibaljhamil@gmail.com"
 *         Apoderado: "Ninguno"
 *         Direccion: "Calle Cantuta # 124"
 */
/**
 * @swagger
 * tags:
 *   name: Cliente
 *   description: API para gestionar Clientes
 */
/**
 * @swagger
 * /incidencias/cliente:
 *   get:
 *     summary: Obtiene la lista de Clientes
 *     tags: [Cliente]
 *     responses:
 *       200:
 *         description: Lista de Clientes.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Cliente'
 */
router.get(
    "/", verifyToken,
    ClienteController.getClienteAll
);
/**
 * @swagger
 * /incidencias/cliente/{id}:
 *   get:
 *     summary: Obtiene un Cliente por ID
 *     tags: [Cliente]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Cliente
 *     responses:
 *       200:
 *         description: Cliente encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Cliente'
 *       404:
 *         description: Cliente no encontrado.
 */
router.get(
    "/:id", verifyToken,
    ClienteController.getClienteById
);
/**
 * @swagger
 * /incidencias/cliente/create:
 *   post:
 *     summary: Crea un nuevo Cliente
 *     tags: [Cliente]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Cliente'
 *     responses:
 *       201:
 *         description: Cliente creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Cliente'
 */
router.post(
    "/create", verifyToken,
    ClienteController.createCliente
);
/**
 * @swagger
 * /incidencias/cliente/update/{id}:
 *   put:
 *     summary: Actualiza un Cliente existente
 *     tags: [Cliente]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Cliente a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Cliente'
 *     responses:
 *       200:
 *         description: Cliente actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Cliente'
 *       404:
 *         description: Cliente no encontrado.
 */
router.put(
    "/update/:id", verifyToken,
    ClienteController.updateCliente
);
/**
 * @swagger
 * /incidencias/cliente/{id}:
 *   delete:
 *     summary: Elimina un Cliente por ID
 *     tags: [Cliente]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Cliente a eliminar
 *     responses:
 *       200:
 *         description: Cliente eliminado exitosamente.
 *       404:
 *         description: Cliente no encontrado.
 */
router.delete(
    "/:id", verifyToken,
    ClienteController.deleteCliente
);

/**
 * @swagger
 * /incidencias/cliente/{id}:
 *   get:
 *     summary: Obtiene un Cliente por ID
 *     tags: [Cliente]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Cliente
 *     responses:
 *       200:
 *         description: Cliente encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Cliente'
 *       404:
 *         description: Cliente no encontrado.
 */
router.get(
    "/search/:documento", verifyToken,
    ClienteController.getClienteByDocumento
);

module.exports = router;
