const express = require("express");
const ConcentimientoController = require("../controllers/concentimiento.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware
// const { validatePerson } = require("../middlewares/validation.middleware");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Concentimiento:
 *       type: object
 *       required:
 *         - RutaDocumento
 *         - TipoConcentimientoId
 *         - PacienteId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Concentimiento
 *         RutaDocumento:
 *           type: string
 *           description: Ruta en la que se obtiene el PDF
 *         TipoConcentimientoId:
 *           type: number
 *           description: NIdentificador del Tipo Concentimiento
 *         PacienteId:
 *           type: number
 *           description: Identificador de la Historia Clinica
 *       example:
 *         id: "1"
 *         RutaDocumento: "f/dasfsd/afds/fa"
 *         TipoConcentimientoId: "1"
 *         PacienteId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Concentimiento
 *   description: API para gestionar Concentimiento
 */
/**
 * @swagger
 * /paciente/concentimiento:
 *   get:
 *     summary: Obtiene la lista de Concentimiento
 *     tags: [Concentimiento]
 *     responses:
 *       200:
 *         description: Lista de Concentimiento.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Concentimiento'
 */
router.get("/", verifyToken, ConcentimientoController.getConcentimientoAll);
/**
 * @swagger
 * /paciente/concentimiento/{id}:
 *   get:
 *     summary: Obtiene un Concentimiento por ID
 *     tags: [Concentimiento]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Concentimiento
 *     responses:
 *       200:
 *         description: Concentimiento encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Concentimiento'
 *       404:
 *         description: Concentimiento no encontrado.
 */
router.get("/:id", verifyToken, ConcentimientoController.getConcentimientoById);
/**
 * @swagger
 * /paciente/concentimiento/create:
 *   post:
 *     summary: Crea un nuevo Concentimiento
 *     tags: [Concentimiento]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Concentimiento'
 *     responses:
 *       201:
 *         description: Concentimiento creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Concentimiento'
 */
router.post("/create", verifyToken, ConcentimientoController.createConcentimiento); // Aplicamos validación
/**
 * @swagger
 * /paciente/concentimiento/update/{id}:
 *   put:
 *     summary: Actualiza un Concentimiento existente
 *     tags: [Concentimiento]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Concentimiento a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Concentimiento'
 *     responses:
 *       200:
 *         description: Concentimiento actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Concentimiento'
 *       404:
 *         description: Concentimiento no encontrado.
 */
router.put("/update/:id", verifyToken, ConcentimientoController.updateConcentimiento);
/**
 * @swagger
 * /paciente/concentimiento/{id}:
 *   delete:
 *     summary: Elimina un Concentimiento por ID
 *     tags: [Concentimiento]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Concentimiento a eliminar
 *     responses:
 *       200:
 *         description: Concentimiento eliminado exitosamente.
 *       404:
 *         description: Concentimiento no encontrado.
 */
router.delete("/:id", verifyToken, ConcentimientoController.deleteConcentimiento);
// 🔍 Rutas individuales para filtrado
/**
 * @swagger
 * /paciente/concentimiento/filtrar/paciente/{paciente}:
 *   get:
 *     summary: Obtiene un Concentimiento por paciente
 *     tags: [Concentimiento]
 *     parameters:
 *       - in: path
 *         name: paciente
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del paciente
 *     responses:
 *       200:
 *         description: Concentimiento encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Concentimiento'
 *       404:
 *         description: Concentimiento no encontrado.
 */
// ruta para filtrar las citas del paciente
router.get("/filtrar/paciente/:paciente", verifyToken, ConcentimientoController.getConcentimientoByPaciente);

module.exports = router;