const express = require("express");
const consultorioController = require("../controllers/consultorio.controller");
const { verifyToken } = require('../middlewares/auth.middleware');

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Consultorio:
 *       type: object
 *       required:
 *         - Nombre
 *         - FotoConsultorio
 *         - ServicioId
 *         - SedeId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del consultorio
 *         Nombre:
 *           type: string
 *           description: Nombre del Consultorio
 *         FotoConsultorio:
 *           type: string
 *           description: Foto del Consultorio
 *         ServicioId:
 *           type: number
 *           description: Servicio proporcionado en el consultorio
 *         SedeId:
 *           type: number
 *           description: El Consultorio pertenece a la sede
 *       example:
 *         id: "1"
 *         Nombre: "Nombre Consultorio"
 *         FotoConsultorio: "url de la foto"
 *         ServicioId: "1"
 *         SedeId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Consultorio
 *   description: API para gestionar Consultorios
 */
/**
 * @swagger
 * /pilares/consultorio:
 *   get:
 *     summary: Obtiene la lista de Consultorios
 *     tags: [Consultorio]
 *     responses:
 *       200:
 *         description: Lista de Consultorios.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Consultorio'
 */
router.get(
    "/", verifyToken,
    consultorioController.getConsultorioAll
);
/**
 * @swagger
 * /pilares/consultorio/sede/{sede}:
 *   get:
 *     summary: Obtiene un Consultorio por la Sede
 *     tags: [Consultorio]
 *     parameters:
 *       - in: path
 *         name: sede
 *         schema:
 *           type: number
 *         required: true
 *         description: ID de la Sede
 *     responses:
 *       200:
 *         description: Consultorios encontrados.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Consultorio'
 *       404:
 *         description: Consultorios no encontrados.
 */
router.get(
    "/sede/:sede", verifyToken,
    consultorioController.getConsultorioBySede
);
/**
 * @swagger
 * /pilares/consultorio/{id}:
 *   get:
 *     summary: Obtiene un Consultorio por ID
 *     tags: [Consultorio]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Consultorio
 *     responses:
 *       200:
 *         description: Consultorio encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Consultorio'
 *       404:
 *         description: Consultorio no encontrado.
 */
router.get(
    "/:id", verifyToken,
    consultorioController.getConsultorioById
);
/**
 * @swagger
 * /pilares/consultorio/create:
 *   post:
 *     summary: Crea un nuevo Consultorio
 *     tags: [Consultorio]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Consultorio'
 *     responses:
 *       201:
 *         description: Consultorio creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Consultorio'
 */
router.post(
    "/create", verifyToken,
    consultorioController.createConsultorio
);
/**
 * @swagger
 * /pilares/consultorio/update/{id}:
 *   put:
 *     summary: Actualiza un Consultorio existente
 *     tags: [Consultorio]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Consultorio a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Consultorio'
 *     responses:
 *       200:
 *         description: Consultorio actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Consultorio'
 *       404:
 *         description: Consultorio no encontrado.
 */
router.put(
    "/update/:id", verifyToken,
    consultorioController.updateConsultorio
);
/**
 * @swagger
 * /pilares/consultorio/{id}:
 *   delete:
 *     summary: Elimina un Consultorio por ID
 *     tags: [Consultorio]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Consultorio a eliminar
 *     responses:
 *       200:
 *         description: Consultorio eliminado exitosamente.
 *       404:
 *         description: Consultorio no encontrado.
 */
router.delete(
    "/:id", verifyToken,
    consultorioController.deleteConsultorio
);

// Rutas de filtrado individual


module.exports = router;
