const express = require("express");
const CronologiaController = require("../controllers/cronologia.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware
// const { validatePerson } = require("../middlewares/validation.middleware");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Cronologia:
 *       type: object
 *       required:
 *         - FechaCronologia
 *         - Descripcion
 *         - ProcedimientoId
 *         - CodigoCIE10P
 *         - DiagnosticoCIE10P
 *         - CodigoCIE10D
 *         - DiagnosticoCIE10D
 *         - AgendamientoId
 *         - PacienteId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Cronologia
 *         FechaCronologia:
 *           type: Date
 *           description: Nombre del Cronologia
 *         Descripcion:
 *           type: string
 *           description: Descripcion de la Cronologia
 *         ProcedimientoId:
 *           type: string
 *           description: Identificador de Procedimiento
 *         CodigoCIE10P:
 *           type: string
 *           description: Codigo CIE10 Presuntivo
 *         DiagnosticoCIE10P:
 *           type: string
 *           description: Diagnostico Presuntivo de la Cronologia
 *         CodigoCIE10D:
 *           type: number
 *           description: Codigo CIE10 Definitivo
 *         DiagnosticoCIE10D:
 *           type: string
 *           description: Diagnostico Definitivo de la Cronologia
 *         AgendamientoId:
 *           type: number
 *           description: Identificador de Agendamiento
 *         PacienteId:
 *           type: number
 *           description: Identificador de la Historia Clinica
 *       example:
 *         id: "1"
 *         FechaCronologia: "2025-11-11"
 *         Descripcion: "gfdsgsg"
 *         ProcedimientoId: "1"
 *         CodigoCIE10P: "gds32"
 *         DiagnosticoCIE10P: "gsdfgfd gfdgdfgdf gfdsgsdf"
 *         CodigoCIE10D: "gsdf23"
 *         DiagnosticoCIE10D: "gfsdg dfgdfsdsgssfdgsdfs fdsgdfsg"
 *         AgendamientoId: "1"
 *         PacienteId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Cronologia
 *   description: API para gestionar Cronologia
 */
/**
 * @swagger
 * /paciente/cronologia:
 *   get:
 *     summary: Obtiene la lista de Cronologia
 *     tags: [Cronologia]
 *     responses:
 *       200:
 *         description: Lista de Cronologia.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Cronologia'
 */
router.get("/", verifyToken, CronologiaController.getCronologiaAll);
/**
 * @swagger
 * /paciente/cronologia/{id}:
 *   get:
 *     summary: Obtiene un Cronologia por ID
 *     tags: [Cronologia]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Cronologia
 *     responses:
 *       200:
 *         description: Cronologia encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Cronologia'
 *       404:
 *         description: Cronologia no encontrado.
 */
router.get("/:id", verifyToken, CronologiaController.getCronologiaById);
/**
 * @swagger
 * /paciente/cronologia/create:
 *   post:
 *     summary: Crea un nuevo Cronologia
 *     tags: [Cronologia]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Cronologia'
 *     responses:
 *       201:
 *         description: Cronologia creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Cronologia'
 */
router.post("/create", verifyToken, CronologiaController.createCronologia); // Aplicamos validación
/**
 * @swagger
 * /paciente/cronologia/update/{id}:
 *   put:
 *     summary: Actualiza un Cronologia existente
 *     tags: [Cronologia]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Cronologia a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Cronologia'
 *     responses:
 *       200:
 *         description: Cronologia actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Cronologia'
 *       404:
 *         description: Cronologia no encontrado.
 */
router.put("/update/:id", verifyToken, CronologiaController.updateCronologia);
/**
 * @swagger
 * /paciente/cronologia/{id}:
 *   delete:
 *     summary: Elimina un Cronologia por ID
 *     tags: [Cronologia]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Cronologia a eliminar
 *     responses:
 *       200:
 *         description: Cronologia eliminado exitosamente.
 *       404:
 *         description: Cronologia no encontrado.
 */
router.delete("/:id", verifyToken, CronologiaController.deleteCronologia);
// 🔍 Rutas individuales para filtrado
/**
 * @swagger
 * /paciente/cronologia/filtrar/paciente/{paciente}:
 *   get:
 *     summary: Obtiene un Cronologia por paciente
 *     tags: [Cronologia]
 *     parameters:
 *       - in: path
 *         name: paciente
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del paciente
 *     responses:
 *       200:
 *         description: Cronologia encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Cronologia'
 *       404:
 *         description: Cronologia no encontrado.
 */
// ruta para filtrar las citas del paciente
router.get("/filtrar/paciente/:paciente", verifyToken, CronologiaController.getCronologiaByPaciente);

module.exports = router;