const express = require("express");
const DienteController = require("../controllers/diente.controller");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Diente:
 *       type: object
 *       required:
 *         - Nombre
 *         - NumeroDental
 *         - UbicacionId
 *         - CuadranteId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Diente
 *         Nombre:
 *           type: string
 *           description: Nombre del Diente
 *         NumeroDental:
 *           type: string
 *           description: Numero Dental del Diente
 *         UbicacionId:
 *           type: string
 *           description: Identificador de Ubicacion
 *         CuadranteId:
 *           type: string
 *           description: Identificador de Cuadrante
 *       example:
 *         id: "1"
 *         Nombre: "FSDASGG dsfdsfsdf"
 *         NumeroDental: "23"
 *         UbicacionId: "1"
 *         CuadranteId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Diente
 *   description: API para gestionar Diente
 */
/**
 * @swagger
 * /odontograma/diente:
 *   get:
 *     summary: Obtiene la lista de Diente
 *     tags: [Diente]
 *     responses:
 *       200:
 *         description: Lista de Diente.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Diente'
 */
router.get("/", DienteController.getDienteAll);
/**
 * @swagger
 * /odontograma/diente/{id}:
 *   get:
 *     summary: Obtiene un Diente por ID
 *     tags: [Diente]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Diente
 *     responses:
 *       200:
 *         description: Diente encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Diente'
 *       404:
 *         description: Diente no encontrado.
 */
router.get("/:id", DienteController.getDienteById);
/**
 * @swagger
 * /odontograma/diente/create:
 *   post:
 *     summary: Crea un nuevo Diente
 *     tags: [Diente]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Diente'
 *     responses:
 *       201:
 *         description: Diente creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Diente'
 */
router.post("/create", DienteController.createDiente);
/**
 * @swagger
 * /odontograma/diente/update/{id}:
 *   put:
 *     summary: Actualiza un Diente existente
 *     tags: [Diente]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Diente a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Diente'
 *     responses:
 *       200:
 *         description: Diente actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Diente'
 *       404:
 *         description: Diente no encontrado.
 */
router.put("/update/:id", DienteController.updateDiente);
/**
 * @swagger
 * /odontograma/diente/{id}:
 *   delete:
 *     summary: Elimina un Diente por ID
 *     tags: [Diente]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Diente a eliminar
 *     responses:
 *       200:
 *         description: Diente eliminado exitosamente.
 *       404:
 *         description: Diente no encontrado.
 */
router.delete("/:id", DienteController.deleteDiente);

module.exports = router;
