const express = require("express");
const DienteClinicoController = require("../controllers/dienteclinico.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware
// const { validatePerson } = require("../middlewares/validation.middleware");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     DienteClinico:
 *       type: object
 *       required:
 *         - Nombre
 *         - RutaImagen
 *         - Procedimiento
 *         - Pieza
 *         - Costo
 *         - CatalogoId
 *         - DienteId
 *         - OdontogramaId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del DienteClinico
 *         Nombre:
 *           type: Date
 *           description: Nombre del DienteClinico
 *         RutaImagen:
 *           type: Date
 *           description: Ruta Imagen del Diente
 *         Procedimiento:
 *           type: string
 *           description: Procedimiento del DienteClinico
 *         Pieza:
 *           type: string
 *           description: Pieza del DienteClinico
 *         Costo:
 *           type: number
 *           description: Costo del DienteClinico
 *         CatalogoId:
 *           type: number
 *           description: Identificador del Catalogo
 *         DienteId:
 *           type: number
 *           description: Identificador del Diente
 *         OdontogramaId:
 *           type: number
 *           description: Identificador del Odontograma
 *       example:
 *         id: "1"
 *         Nombre: "asgasd asdgdsgds sdgasdgasdg dsgsagd"
 *         RutaImagen: "fadsf/fadfsda/gadsgsad.fd"
 *         Procedimiento: "fdsafsd fadsdasgads asdgasdgads"
 *         Pieza: "32"
 *         Costo: "120"
 *         CatalogoId: "1"
 *         DienteId: "1"
 *         OdontogramaId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: DienteClinico
 *   description: API para gestionar DienteClinico
 */
/**
 * @swagger
 * /odontograma/DienteClinico:
 *   get:
 *     summary: Obtiene la lista de DienteClinico
 *     tags: [DienteClinico]
 *     responses:
 *       200:
 *         description: Lista de DienteClinico.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/DienteClinico'
 */
router.get("/", verifyToken, DienteClinicoController.getDienteClinicoAll);
/**
 * @swagger
 * /odontograma/DienteClinico/{id}:
 *   get:
 *     summary: Obtiene un DienteClinico por ID
 *     tags: [DienteClinico]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del DienteClinico
 *     responses:
 *       200:
 *         description: DienteClinico encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/DienteClinico'
 *       404:
 *         description: DienteClinico no encontrado.
 */
router.get("/:id", verifyToken, DienteClinicoController.getDienteClinicoById);
/**
 * @swagger
 * /odontograma/DienteClinico/create:
 *   post:
 *     summary: Crea un nuevo DienteClinico
 *     tags: [DienteClinico]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/DienteClinico'
 *     responses:
 *       201:
 *         description: DienteClinico creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/DienteClinico'
 */
router.post("/create", verifyToken, DienteClinicoController.createDienteClinico); // Aplicamos validación
/**
 * @swagger
 * /odontograma/DienteClinico/update/{id}:
 *   put:
 *     summary: Actualiza un DienteClinico existente
 *     tags: [DienteClinico]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del DienteClinico a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/DienteClinico'
 *     responses:
 *       200:
 *         description: DienteClinico actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/DienteClinico'
 *       404:
 *         description: DienteClinico no encontrado.
 */
router.put("/update/:id", verifyToken, DienteClinicoController.updateDienteClinico);
/**
 * @swagger
 * /odontograma/DienteClinico/{id}:
 *   delete:
 *     summary: Elimina un DienteClinico por ID
 *     tags: [DienteClinico]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del DienteClinico a eliminar
 *     responses:
 *       200:
 *         description: DienteClinico eliminado exitosamente.
 *       404:
 *         description: DienteClinico no encontrado.
 */
router.delete("/:id", verifyToken, DienteClinicoController.deleteDienteClinico);
// 🔍 Rutas individuales para filtrado
/**
 * @swagger
 * /odontograma/DienteClinico/filtrar/odontograma/{odontograma}:
 *   get:
 *     summary: Obtiene un DienteClinico por odontograma
 *     tags: [DienteClinico]
 *     parameters:
 *       - in: path
 *         name: odontograma
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del odontograma
 *     responses:
 *       200:
 *         description: DienteClinico encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/DienteClinico'
 *       404:
 *         description: DienteClinico no encontrado.
 */
// ruta para filtrar las citas del paciente
router.get("/filtrar/odontograma/:odontograma", verifyToken, DienteClinicoController.getDienteClinicoByOdontograma);

module.exports = router;