const express = require("express");
const DienteEconomicoController = require("../controllers/dienteeconomico.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware
// const { validatePerson } = require("../middlewares/validation.middleware");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     DienteEconomico:
 *       type: object
 *       required:
 *         - Nombre
 *         - RutaImagen
 *         - Procedimiento
 *         - Pieza
 *         - Costo
 *         - CatalogoId
 *         - DienteId
 *         - OdontogramaId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del DienteEconomico
 *         Nombre:
 *           type: Date
 *           description: Nombre del DienteEconomico
 *         RutaImagen:
 *           type: Date
 *           description: Ruta Imagen del Diente
 *         Procedimiento:
 *           type: string
 *           description: Procedimiento del DienteEconomico
 *         Pieza:
 *           type: string
 *           description: Pieza del DienteEconomico
 *         Costo:
 *           type: number
 *           description: Costo del DienteEconomico
 *         CatalogoId:
 *           type: number
 *           description: Identificador del Catalogo
 *         DienteId:
 *           type: number
 *           description: Identificador del Diente
 *         OdontogramaId:
 *           type: number
 *           description: Identificador del Odontograma
 *       example:
 *         id: "1"
 *         Nombre: "asgasd asdgdsgds sdgasdgasdg dsgsagd"
 *         RutaImagen: "fadsf/fadfsda/gadsgsad.fd"
 *         Procedimiento: "fdsafsd fadsdasgads asdgasdgads"
 *         Pieza: "32"
 *         Costo: "120"
 *         CatalogoId: "1"
 *         DienteId: "1"
 *         OdontogramaId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: DienteEconomico
 *   description: API para gestionar DienteEconomico
 */
/**
 * @swagger
 * /odontograma/DienteEconomico:
 *   get:
 *     summary: Obtiene la lista de DienteEconomico
 *     tags: [DienteEconomico]
 *     responses:
 *       200:
 *         description: Lista de DienteEconomico.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/DienteEconomico'
 */
router.get("/", verifyToken, DienteEconomicoController.getDienteEconomicoAll);
/**
 * @swagger
 * /odontograma/DienteEconomico/{id}:
 *   get:
 *     summary: Obtiene un DienteEconomico por ID
 *     tags: [DienteEconomico]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del DienteEconomico
 *     responses:
 *       200:
 *         description: DienteEconomico encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/DienteEconomico'
 *       404:
 *         description: DienteEconomico no encontrado.
 */
router.get("/:id", verifyToken, DienteEconomicoController.getDienteEconomicoById);
/**
 * @swagger
 * /odontograma/DienteEconomico/create:
 *   post:
 *     summary: Crea un nuevo DienteEconomico
 *     tags: [DienteEconomico]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/DienteEconomico'
 *     responses:
 *       201:
 *         description: DienteEconomico creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/DienteEconomico'
 */
router.post("/create", verifyToken, DienteEconomicoController.createDienteEconomico); // Aplicamos validación
/**
 * @swagger
 * /odontograma/DienteEconomico/update/{id}:
 *   put:
 *     summary: Actualiza un DienteEconomico existente
 *     tags: [DienteEconomico]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del DienteEconomico a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/DienteEconomico'
 *     responses:
 *       200:
 *         description: DienteEconomico actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/DienteEconomico'
 *       404:
 *         description: DienteEconomico no encontrado.
 */
router.put("/update/:id", verifyToken, DienteEconomicoController.updateDienteEconomico);
/**
 * @swagger
 * /odontograma/DienteEconomico/{id}:
 *   delete:
 *     summary: Elimina un DienteEconomico por ID
 *     tags: [DienteEconomico]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del DienteEconomico a eliminar
 *     responses:
 *       200:
 *         description: DienteEconomico eliminado exitosamente.
 *       404:
 *         description: DienteEconomico no encontrado.
 */
router.delete("/:id", verifyToken, DienteEconomicoController.deleteDienteEconomico);
// 🔍 Rutas individuales para filtrado
/**
 * @swagger
 * /odontograma/DienteEconomico/filtrar/odontograma/{odontograma}:
 *   get:
 *     summary: Obtiene un DienteEconomico por odontograma
 *     tags: [DienteEconomico]
 *     parameters:
 *       - in: path
 *         name: odontograma
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del odontograma
 *     responses:
 *       200:
 *         description: DienteEconomico encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/DienteEconomico'
 *       404:
 *         description: DienteEconomico no encontrado.
 */
// ruta para filtrar las citas del paciente
router.get("/filtrar/odontograma/:odontograma", verifyToken, DienteEconomicoController.getDienteEconomicoByOdontograma);

module.exports = router;