const express = require("express");
const documentoController = require("../controllers/documento.controller");
const { verifyToken } = require('../middlewares/auth.middleware');

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Documento:
 *       type: object
 *       required:
 *         - NombreArchivo
 *         - DescripcionBreve
 *         - FechaSubida
 *         - RutaArchivo
 *         - TipoDocPDFId
 *         - UsuarioId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Documento
 *         NombreArchivo:
 *           type: string
 *           description: Nombre del Archivo PDF que se subio con una normalizacion
 *         DescripcionBreve:
 *           type: string
 *           description: Descripcion breve de que se trata el documento subido
 *         FechaSubida:
 *           type: Date
 *           description: Fecha de Subida del Archivo PDF
 *         RutaArchivo:
 *           type: string
 *           description: Ruta en la que se sube el PDF, para luego poder descargar
 *         TipoDocPDFId:
 *           type: number
 *           description: Que tipo de Doc se sube (Titulo, Certificado, Maestria, Doctorado, Especialidad, etc)
 *         UsuarioId:
 *           type: number
 *           description: Trabajador al que le pertenece el Archivo
 *       example:
 *         id: "1"
 *         NombreArchivo: "Titulo del trabajador"
 *         DescripcionBreve: "Titulo de Administrador"
 *         FechaSubida: "2021-09-01"
 *         RutaArchivo: "./uploads/Titulo/mititulo.pdf"
 *         TipoDocPDFId: "1"
 *         UsuarioId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Documento
 *   description: API para gestionar Documentos
 */
/**
 * @swagger
 * /rrhh/documento:
 *   get:
 *     summary: Obtiene la lista de Documentos
 *     tags: [Documento]
 *     responses:
 *       200:
 *         description: Lista de Documentos.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Documento'
 */
router.get("/", verifyToken, documentoController.getDocumentoAll);
/**
 * @swagger
 * /rrhh/documento/trabajador/{trabajador}:
 *   get:
 *     summary: Obtiene Documentos por trabajador
 *     tags: [Documento]
 *     parameters:
 *       - in: path
 *         name: trabajador
 *         schema:
 *           type: number
 *         required: true
 *         description: documentos del trabajador
 *     responses:
 *       200:
 *         description: Documentos encontrados.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Documento'
 *       404:
 *         description: Documento no encontrado.
 */
router.get("/trabajador/:trabajador", verifyToken, documentoController.getDocumentoByTrabajador);
/**
 * @swagger
 * /rrhh/documento/{id}:
 *   get:
 *     summary: Obtiene un Documento por ID
 *     tags: [Documento]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Documento
 *     responses:
 *       200:
 *         description: Documento encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Documento'
 *       404:
 *         description: Documento no encontrado.
 */
router.get("/:id", verifyToken, documentoController.getDocumentoById);
/**
 * @swagger
 * /rrhh/documento/create:
 *   post:
 *     summary: Crea un nuevo Documento
 *     tags: [Documento]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Documento'
 *     responses:
 *       201:
 *         description: Documento creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Documento'
 */
router.post("/create", verifyToken, documentoController.createDocumento);
/**
 * @swagger
 * /rrhh/documento/update/{id}:
 *   put:
 *     summary: Actualiza un Documento existente
 *     tags: [Documento]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Documento a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Documento'
 *     responses:
 *       200:
 *         description: Documento actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Documento'
 *       404:
 *         description: Documento no encontrado.
 */
router.put("/update/:id", verifyToken, documentoController.updateDocumento);
/**
 * @swagger
 * /rrhh/documento/{id}:
 *   delete:
 *     summary: Elimina un Documento por ID
 *     tags: [Documento]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Documento a eliminar
 *     responses:
 *       200:
 *         description: Documento eliminado exitosamente.
 *       404:
 *         description: Documento no encontrado.
 */
router.delete("/:id", verifyToken, documentoController.deleteDocumento);

// Rutas de filtrado individual


module.exports = router;
