const express = require("express");
const EgresoController = require("../controllers/egreso.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Egreso:
 *       type: object
 *       required:
 *         - Nombre
 *         - Proveedor
 *         - Detalles
 *         - NroFactura
 *         - MontoSoles
 *         - MontoDolares
 *         - SedeId
 *         - UsuarioId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Egreso
 *         Nombre:
 *           type: string
 *           description: Nombre del Egreso
 *         Proveedor:
 *           type: string
 *           description: Proveedor del Egreso
 *         Detalles:
 *           type: string
 *           description: Detalles del Egreso
 *         NroFactura:
 *           type: string
 *           description: NroFactura del Egreso
 *         MontoSoles:
 *           type: string
 *           description: Monto Soles del Egreso
 *         MontoDolares:
 *           type: string
 *           description: Monto Dolares del Egreso
 *         SedeId:
 *           type: string
 *           description: Sede del Egreso
 *         UsuarioId:
 *           type: string
 *           description: Usuario del Egreso
 *       example:
 *         id: "1"
 *         Nombre: "Calle Cantuta 124"
 *         Proveedor: "Nombre Servicio, Personal, etc"
 *         Detalles: "gdfgdfsg dfgdfg dfs"
 *         NroFactura: "bret34543trehdf"
 *         MontoSoles: "100"
 *         MontoDolares: "30"
 *         SedeId: "1"
 *         UsuarioId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Egreso
 *   description: API para gestionar Egreso
 */
/**
 * @swagger
 * /economia/egreso:
 *   get:
 *     summary: Obtiene la lista de Egreso
 *     tags: [Egreso]
 *     responses:
 *       200:
 *         description: Lista de Egreso.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Egreso'
 */
router.get("/", verifyToken, EgresoController.getEgresoAll);
/**
 * @swagger
 * /economia/egreso/{id}:
 *   get:
 *     summary: Obtiene un Egreso por ID
 *     tags: [Egreso]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Egreso
 *     responses:
 *       200:
 *         description: Egreso encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Egreso'
 *       404:
 *         description: Egreso no encontrado.
 */
router.get("/:id", verifyToken, EgresoController.getEgresoById);
/**
 * @swagger
 * /economia/egreso/create:
 *   post:
 *     summary: Crea un nuevo Egreso
 *     tags: [Egreso]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Egreso'
 *     responses:
 *       201:
 *         description: Egreso creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Egreso'
 */
router.post("/create", verifyToken, EgresoController.createEgreso);
/**
 * @swagger
 * /economia/egreso/update/{id}:
 *   put:
 *     summary: Actualiza un Egreso existente
 *     tags: [Egreso]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Egreso a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Egreso'
 *     responses:
 *       200:
 *         description: Egreso actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Egreso'
 *       404:
 *         description: Egreso no encontrado.
 */
router.put("/update/:id", verifyToken, EgresoController.updateEgreso);
/**
 * @swagger
 * /economia/egreso/{id}:
 *   delete:
 *     summary: Elimina un Egreso por ID
 *     tags: [Egreso]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Egreso a eliminar
 *     responses:
 *       200:
 *         description: Egreso eliminado exitosamente.
 *       404:
 *         description: Egreso no encontrado.
 */
router.delete("/:id", verifyToken, EgresoController.deleteEgreso);
/**
 * @swagger
 * /economia/egreso/filtrar/sede/{sede}:
 *   get:
 *     summary: Obtiene un Egreso por sede
 *     tags: [Egreso]
 *     parameters:
 *       - in: path
 *         name: sede
 *         schema:
 *           type: number
 *         required: true
 *         description: sede del Egreso
 *     responses:
 *       200:
 *         description: Egreso encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Egreso'
 *       404:
 *         description: Egreso no encontrado.
 */
router.get("/filtrar/sede/:sede", verifyToken, EgresoController.getEgresoBySede);

module.exports = router;
