const express = require("express");
const EmitirBoletaController = require("../controllers/emisionboleta.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     EmitirBoleta:
 *       type: object
 *       required:
 *         - Descripcion
 *         - SubTotal
 *         - IGV
 *         - Total
 *         - FacturaId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Emitir Boleta
 *         Descripcion:
 *           type: string
 *           description: Descripcion de la Emision
 *         SubTotal:
 *           type: number
 *           description: SubTotal de la Emision
 *         IGV:
 *           type: number
 *           description: IGV de la Emision
 *         Total:
 *           type: number
 *           description: Total de la Emision
 *         FacturaId:
 *           type: number
 *           description: Identifiador de la Factura
 *       example:
 *         id: "1"
 *         Descripcion: "fdsfasdgasdg"
 *         SubTotal: "86"
 *         IGV: "18"
 *         Total: "100"
 *         FacturaId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: EmitirBoleta
 *   description: API para gestionar EmitirBoleta
 */
/**
 * @swagger
 * /economia/emitirboleta:
 *   get:
 *     summary: Obtiene la lista de EmitirBoleta
 *     tags: [EmitirBoleta]
 *     responses:
 *       200:
 *         description: Lista de EmitirBoleta.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/EmitirBoleta'
 */
router.get("/", verifyToken, EmitirBoletaController.getEmitirBoletaAll);
/**
 * @swagger
 * /economia/emitirboleta/{id}:
 *   get:
 *     summary: Obtiene un EmitirBoleta por ID
 *     tags: [EmitirBoleta]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del EmitirBoleta
 *     responses:
 *       200:
 *         description: EmitirBoleta encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/EmitirBoleta'
 *       404:
 *         description: EmitirBoleta no encontrado.
 */
router.get("/:id", verifyToken, EmitirBoletaController.getEmitirBoletaById);
/**
 * @swagger
 * /economia/emitirboleta/create:
 *   post:
 *     summary: Crea un nuevo EmitirBoleta
 *     tags: [EmitirBoleta]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/EmitirBoleta'
 *     responses:
 *       201:
 *         description: EmitirBoleta creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/EmitirBoleta'
 */
router.post("/create", verifyToken, EmitirBoletaController.createEmitirBoleta);
/**
 * @swagger
 * /economia/emitirboleta/update/{id}:
 *   put:
 *     summary: Actualiza un EmitirBoleta existente
 *     tags: [EmitirBoleta]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del EmitirBoleta a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/EmitirBoleta'
 *     responses:
 *       200:
 *         description: EmitirBoleta actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/EmitirBoleta'
 *       404:
 *         description: EmitirBoleta no encontrado.
 */
router.put("/update/:id", verifyToken, EmitirBoletaController.updateEmitirBoleta);
/**
 * @swagger
 * /economia/emitirboleta/{id}:
 *   delete:
 *     summary: Elimina un EmitirBoleta por ID
 *     tags: [EmitirBoleta]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del EmitirBoleta a eliminar
 *     responses:
 *       200:
 *         description: EmitirBoleta eliminado exitosamente.
 *       404:
 *         description: EmitirBoleta no encontrado.
 */
router.delete("/:id", verifyToken, EmitirBoletaController.deleteEmitirBoleta);
/**
 * @swagger
 * /economia/emitirboleta/filtrar/factura/{factura}:
 *   get:
 *     summary: Obtiene un EmitirBoleta por usuario
 *     tags: [EmitirBoleta]
 *     parameters:
 *       - in: path
 *         name: factura
 *         schema:
 *           type: number
 *         required: true
 *         description: factura del EmitirBoleta
 *     responses:
 *       200:
 *         description: EmitirBoleta encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/EmitirBoleta'
 *       404:
 *         description: EmitirBoleta no encontrado.
 */
router.get("/filtrar/factura/:factura", verifyToken, EmitirBoletaController.getEmitirBoletaByFactura);

module.exports = router;
