const express = require("express");
const EmitirFacturaController = require("../controllers/emisionfactura.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     EmitirFactura:
 *       type: object
 *       required:
 *         - Descripcion
 *         - SubTotal
 *         - IGV
 *         - Total
 *         - FacturaId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Emitir Factura
 *         Descripcion:
 *           type: string
 *           description: Descripcion de la Emision
 *         SubTotal:
 *           type: number
 *           description: SubTotal de la Emision
 *         IGV:
 *           type: number
 *           description: IGV de la Emision
 *         Total:
 *           type: number
 *           description: Total de la Emision
 *         FacturaId:
 *           type: number
 *           description: Identifiador de la Factura
 *       example:
 *         id: "1"
 *         Descripcion: "fdsfasdgasdg"
 *         SubTotal: "86"
 *         IGV: "18"
 *         Total: "100"
 *         FacturaId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: EmitirFactura
 *   description: API para gestionar EmitirFactura
 */
/**
 * @swagger
 * /economia/emitirfactura:
 *   get:
 *     summary: Obtiene la lista de EmitirFactura
 *     tags: [EmitirFactura]
 *     responses:
 *       200:
 *         description: Lista de EmitirFactura.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/EmitirFactura'
 */
router.get("/", verifyToken, EmitirFacturaController.getEmitirFacturaAll);
/**
 * @swagger
 * /economia/emitirfactura/{id}:
 *   get:
 *     summary: Obtiene un EmitirFactura por ID
 *     tags: [EmitirFactura]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del EmitirFactura
 *     responses:
 *       200:
 *         description: EmitirFactura encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/EmitirFactura'
 *       404:
 *         description: EmitirFactura no encontrado.
 */
router.get("/:id", verifyToken, EmitirFacturaController.getEmitirFacturaById);
/**
 * @swagger
 * /economia/emitirfactura/create:
 *   post:
 *     summary: Crea un nuevo EmitirFactura
 *     tags: [EmitirFactura]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/EmitirFactura'
 *     responses:
 *       201:
 *         description: EmitirFactura creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/EmitirFactura'
 */
router.post("/create", verifyToken, EmitirFacturaController.createEmitirFactura);
/**
 * @swagger
 * /economia/emitirfactura/update/{id}:
 *   put:
 *     summary: Actualiza un EmitirFactura existente
 *     tags: [EmitirFactura]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del EmitirFactura a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/EmitirFactura'
 *     responses:
 *       200:
 *         description: EmitirFactura actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/EmitirFactura'
 *       404:
 *         description: EmitirFactura no encontrado.
 */
router.put("/update/:id", verifyToken, EmitirFacturaController.updateEmitirFactura);
/**
 * @swagger
 * /economia/emitirfactura/{id}:
 *   delete:
 *     summary: Elimina un EmitirFactura por ID
 *     tags: [EmitirFactura]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del EmitirFactura a eliminar
 *     responses:
 *       200:
 *         description: EmitirFactura eliminado exitosamente.
 *       404:
 *         description: EmitirFactura no encontrado.
 */
router.delete("/:id", verifyToken, EmitirFacturaController.deleteEmitirFactura);
/**
 * @swagger
 * /economia/emitirfactura/filtrar/factura/{factura}:
 *   get:
 *     summary: Obtiene un EmitirFactura por usuario
 *     tags: [EmitirFactura]
 *     parameters:
 *       - in: path
 *         name: factura
 *         schema:
 *           type: number
 *         required: true
 *         description: factura del EmitirFactura
 *     responses:
 *       200:
 *         description: EmitirFactura encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/EmitirFactura'
 *       404:
 *         description: EmitirFactura no encontrado.
 */
router.get("/filtrar/factura/:factura", verifyToken, EmitirFacturaController.getEmitirFacturaByFactura);

module.exports = router;
