const express = require("express");
const empresaController = require("../controllers/empresa.controller");
const { verifyToken } = require('../middlewares/auth.middleware');

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Empresa:
 *       type: object
 *       required:
 *         - Nombre
 *         - RazonSocial
 *         - NombreComercial
 *         - RUC
 *         - TipoEmpresa
 *         - Sector
 *         - SitioWeb
 *         - Facebook
 *         - Instagram
 *         - TikTok
 *         - YouTube
 *         - Correo
 *         - Telefono
 *         - Celular
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único de la Empresa
 *         RazonSocial:
 *           type: string
 *           description: Nombre Registrado en la Sunat
 *         NombreComercial:
 *           type: string
 *           description: Como su publico lo conoce
 *         RUC:
 *           type: string
 *           description: Identificador en la SUNAT
 *         TipoEmpresa:
 *           type: string
 *           description: Publica o Privada
 *         Sector:
 *           type: string
 *           description: A que sector de comercio se dedica
 *         SitioWeb:
 *           type: string
 *           description: la url de su pagina web
 *         Facebook:
 *           type: string
 *           description: la url de su fan page de facebook
 *         Instagram:
 *           type: string
 *           description: la url de como se encuentra en instagram
 *         TikTok:
 *           type: string
 *           description: la url de como se encuentra en tiktok
 *         YouTube:
 *           type: string
 *           description: la url de como se encuentra en youtube
 *         Correo:
 *           type: string
 *           description: correo principal de la empresa
 *         Telefono:
 *           type: string
 *           description: telefono principal de la empresa
 *         Celular:
 *           type: string
 *           description: celular principal de la empresa
 *       example:
 *         id: "1"
 *         RazonSocial: 'Clinica Dental Imagen SAC'
 *         NombreComercial: "C.O. Imagen"
 *         RUC: "20486238322"
 *         TipoEmpresa: "Privada"
 *         Sector: "Salud"
 *         SitioWeb: "https://clinicadentalimagen.pe/"
 *         Facebook: "https://www.facebook.com/clinicadentalimagen.sac"
 *         Instagram: "https://www.instagram.com/clinicadentalimagenoficial/"
 *         TikTok: "https://www.tiktok.com/@clinicadentalimagen"
 *         YouTube: ""
 *         Correo: "clinicadentalimagensac@gmail.com"
 *         Telefono: ""
 *         Celular: ""
 */
/**
 * @swagger
 * tags:
 *   name: Empresa
 *   description: API para gestionar Empresas
 */
/**
 * @swagger
 * /pilares/Empresa:
 *   get:
 *     summary: Obtiene la lista de Empresas
 *     tags: [Empresa]
 *     responses:
 *       200:
 *         description: Lista de Empresas.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Empresa'
 */
router.get(
    "/", verifyToken,
    empresaController.getEmpresaAll
);
/**
 * @swagger
 * /pilares/Empresa/{id}:
 *   get:
 *     summary: Obtiene un Empresa por ID
 *     tags: [Empresa]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Empresa
 *     responses:
 *       200:
 *         description: Empresa encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Empresa'
 *       404:
 *         description: Empresa no encontrado.
 */
router.get(
    "/:id", verifyToken,
    empresaController.getEmpresaById
);
/**
 * @swagger
 * /pilares/Empresa/create:
 *   post:
 *     summary: Crea un nuevo Empresa
 *     tags: [Empresa]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Empresa'
 *     responses:
 *       201:
 *         description: Empresa creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Empresa'
 */
router.post(
    "/create", verifyToken,
    empresaController.createEmpresa
);
/**
 * @swagger
 * /pilares/Empresa/update/{id}:
 *   put:
 *     summary: Actualiza un Empresa existente
 *     tags: [Empresa]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Empresa a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Empresa'
 *     responses:
 *       200:
 *         description: Empresa actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Empresa'
 *       404:
 *         description: Empresa no encontrado.
 */
router.put(
    "/update/:id", verifyToken,
    empresaController.updateEmpresa
);
/**
 * @swagger
 * /pilares/Empresa/{id}:
 *   delete:
 *     summary: Elimina un Empresa por ID
 *     tags: [Empresa]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Empresa a eliminar
 *     responses:
 *       200:
 *         description: Empresa eliminado exitosamente.
 *       404:
 *         description: Empresa no encontrado.
 */
router.delete(
    "/:id", verifyToken,
    empresaController.deleteEmpresa
);

module.exports = router;
