const express = require("express");
const EnfermedadActualController = require("../controllers/enfermedadactual.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware
// const { validatePerson } = require("../middlewares/validation.middleware");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     EnfermedadActual:
 *       type: object
 *       required:
 *         - MotivoConsulta
 *         - TiempodeEnfermedad
 *         - SignosySintomasPrincipales
 *         - RelatoCronologico
 *         - FuncionesBiologicas
 *         - AgendamientoId
 *         - PacienteId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del EnfermedadActual
 *         MotivoConsulta:
 *           type: string
 *           description: Motivo de la Consulta
 *         TiempodeEnfermedad:
 *           type: string
 *           description: Tiempo de Enfermedad Actual
 *         SignosySintomasPrincipales:
 *           type: string
 *           description: Signos y Sintomas Principales
 *         RelatoCronologico:
 *           type: string
 *           description: Relato Cronologico
 *         FuncionesBiologicas:
 *           type: string
 *           description: Funciones Biologicas
 *         AgendamientoId:
 *           type: number
 *           description: Identificador de Agendamiento
 *         PacienteId:
 *           type: number
 *           description: Identificador de la Historia Clinica
 *       example:
 *         id: "1"
 *         MotivoConsulta: "fasdfasddsg ggfds fgfdsgfdsgff"
 *         TiempodeEnfermedad: "fdsafadssda d fdsafsd"
 *         SignosySintomasPrincipales: "fdsafdasasd dsa sda"
 *         RelatoCronologico: "fdsafdsf ds sdafdsdsdsa"
 *         FuncionesBiologicas: "fdsafsd fsadasdsd"
 *         AgendamientoId: "1"
 *         PacienteId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: EnfermedadActual
 *   description: API para gestionar EnfermedadActual
 */
/**
 * @swagger
 * /paciente/enfermedadactual:
 *   get:
 *     summary: Obtiene la lista de EnfermedadActual
 *     tags: [EnfermedadActual]
 *     responses:
 *       200:
 *         description: Lista de EnfermedadActual.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/EnfermedadActual'
 */
router.get("/", verifyToken, EnfermedadActualController.getEnfermedadActualAll);
/**
 * @swagger
 * /paciente/enfermedadactual/{id}:
 *   get:
 *     summary: Obtiene un EnfermedadActual por ID
 *     tags: [EnfermedadActual]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del EnfermedadActual
 *     responses:
 *       200:
 *         description: EnfermedadActual encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/EnfermedadActual'
 *       404:
 *         description: EnfermedadActual no encontrado.
 */
router.get("/:id", verifyToken, EnfermedadActualController.getEnfermedadActualById);
/**
 * @swagger
 * /paciente/enfermedadactual/create:
 *   post:
 *     summary: Crea un nuevo EnfermedadActual
 *     tags: [EnfermedadActual]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/EnfermedadActual'
 *     responses:
 *       201:
 *         description: EnfermedadActual creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/EnfermedadActual'
 */
router.post("/create", verifyToken, EnfermedadActualController.createEnfermedadActual); // Aplicamos validación
/**
 * @swagger
 * /paciente/enfermedadactual/update/{id}:
 *   put:
 *     summary: Actualiza un EnfermedadActual existente
 *     tags: [EnfermedadActual]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del EnfermedadActual a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/EnfermedadActual'
 *     responses:
 *       200:
 *         description: EnfermedadActual actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/EnfermedadActual'
 *       404:
 *         description: EnfermedadActual no encontrado.
 */
router.put("/update/:id", verifyToken, EnfermedadActualController.updateEnfermedadActual);
/**
 * @swagger
 * /paciente/enfermedadactual/{id}:
 *   delete:
 *     summary: Elimina un EnfermedadActual por ID
 *     tags: [EnfermedadActual]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del EnfermedadActual a eliminar
 *     responses:
 *       200:
 *         description: EnfermedadActual eliminado exitosamente.
 *       404:
 *         description: EnfermedadActual no encontrado.
 */
router.delete("/:id", verifyToken, EnfermedadActualController.deleteEnfermedadActual);
// 🔍 Rutas individuales para filtrado
/**
 * @swagger
 * /paciente/enfermedadactual/filtrar/paciente/{paciente}:
 *   get:
 *     summary: Obtiene un EnfermedadActual por paciente
 *     tags: [EnfermedadActual]
 *     parameters:
 *       - in: path
 *         name: paciente
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del paciente
 *     responses:
 *       200:
 *         description: EnfermedadActual encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/EnfermedadActual'
 *       404:
 *         description: EnfermedadActual no encontrado.
 */
// ruta para filtrar las citas del paciente
router.get("/filtrar/paciente/:paciente", verifyToken, EnfermedadActualController.getEnfermedadActualByPaciente);

module.exports = router;