const express = require("express");
const EstadoDienteController = require("../controllers/estadodiente.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     EstadoDiente:
 *       type: object
 *       required:
 *         - NombreDiagnostico
 *         - Pieza
 *         - Nombre
 *         - RutaImagen
 *         - DienteId
 *         - DiagnosticoId
 *         - OdontogramaId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del EstadoDiente
 *         NombreDiagnostico:
 *           type: Date
 *           description: Nombre Diagnostico del Diente
 *         Pieza:
 *           type: number
 *           description: Numero de Pieza Involucrado
 *         Nombre:
 *           type: string
 *           description: Nombre del EstadoDiente
 *         RutaImagen:
 *           type: string
 *           description: Ruta de la Imagen del Diente
 *         DienteId:
 *           type: number
 *           description: Identificador del Diente
 *         DiagnosticoId:
 *           type: number
 *           description: Identificador del Diagnostico
 *         OdontogramaId:
 *           type: number
 *           description: Identificador del Odontograma
 *       example:
 *         id: "1"
 *         NombreDiagnostico: "fdsagas dasgadsgads sadgsdgsg"
 *         Pieza: "21"
 *         Nombre: "fadsgsadgds"
 *         RutaImagen: "fasdf/fadsfasdf/fadfasf.fasd"
 *         DienteId: "1"
 *         DiagnosticoId: "1"
 *         OdontogramaId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: EstadoDiente
 *   description: API para gestionar EstadoDiente
 */
/**
 * @swagger
 * /odontograma/estadodiente:
 *   get:
 *     summary: Obtiene la lista de EstadoDiente
 *     tags: [EstadoDiente]
 *     responses:
 *       200:
 *         description: Lista de EstadoDiente.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/EstadoDiente'
 */
router.get("/", verifyToken, EstadoDienteController.getEstadoDienteAll);
/**
 * @swagger
 * /odontograma/estadodiente/{id}:
 *   get:
 *     summary: Obtiene un EstadoDiente por ID
 *     tags: [EstadoDiente]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del EstadoDiente
 *     responses:
 *       200:
 *         description: EstadoDiente encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/EstadoDiente'
 *       404:
 *         description: EstadoDiente no encontrado.
 */
router.get("/:id", verifyToken, EstadoDienteController.getEstadoDienteById);
/**
 * @swagger
 * /odontograma/estadodiente/create:
 *   post:
 *     summary: Crea un nuevo EstadoDiente
 *     tags: [EstadoDiente]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/EstadoDiente'
 *     responses:
 *       201:
 *         description: EstadoDiente creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/EstadoDiente'
 */
router.post("/create", verifyToken, EstadoDienteController.createEstadoDiente); // Aplicamos validación
/**
 * @swagger
 * /odontograma/estadodiente/update/{id}:
 *   put:
 *     summary: Actualiza un EstadoDiente existente
 *     tags: [EstadoDiente]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del EstadoDiente a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/EstadoDiente'
 *     responses:
 *       200:
 *         description: EstadoDiente actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/EstadoDiente'
 *       404:
 *         description: EstadoDiente no encontrado.
 */
router.put("/update/:id", verifyToken, EstadoDienteController.updateEstadoDiente);
/**
 * @swagger
 * /odontograma/estadodiente/{id}:
 *   delete:
 *     summary: Elimina un EstadoDiente por ID
 *     tags: [EstadoDiente]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del EstadoDiente a eliminar
 *     responses:
 *       200:
 *         description: EstadoDiente eliminado exitosamente.
 *       404:
 *         description: EstadoDiente no encontrado.
 */
router.delete("/:id", verifyToken, EstadoDienteController.deleteEstadoDiente);
// 🔍 Rutas individuales para filtrado
/**
 * @swagger
 * /odontograma/estadodiente/filtrar/hallazgo/{hallazgo}/{magnitud}/{posicion}:
 *   get:
 *     summary: Obtiene Información de EstadoDiente filtrada por Hallazgo, Magnitud y Posición
 *     tags: [EstadoDiente]
 *     parameters:
 *       - in: path
 *         name: hallazgo
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del Hallazgo
 *       - in: path
 *         name: magnitud
 *         schema:
 *           type: number
 *         required: true
 *         description: Magnitud del Diente (1 -> Grande, 2 -> Pequeño)
 *       - in: path
 *         name: posicion
 *         schema:
 *           type: number
 *         required: true
 *         description: Posicion del Diente (1 -> Superior, 2 -> Inferior)
 *     responses:
 *       200:
 *         description: EstadoDiente encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/EstadoDiente'
 *       404:
 *         description: EstadoDiente no encontrado.
 */
// ruta para filtrar las citas del paciente
router.get("/filtrar/hallazgo/:hallazgo/:magnitud/:posicion", verifyToken, EstadoDienteController.getEstadoDienteByOdontograma);

module.exports = router;