const express = require("express");
const evaluacionController = require("../controllers/evaluacion.controller");
const { verifyToken } = require('../middlewares/auth.middleware');

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Evaluacion:
 *       type: object
 *       required:
 *         - FechaEvaluacion
 *         - Evaluador
 *         - Puntaje
 *         - Comentario
 *         - UsuarioId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único de la Evaluacion
 *         FechaEvaluacion:
 *           type: Date
 *           description: Fecha de Evaluacion
 *         Evaluador:
 *           type: string
 *           description: Quien lo Evaluo
 *         Puntaje:
 *           type: string
 *           description: Puntaje otorgado por el Evaluador
 *         Comentario:
 *           type: string
 *           description: Comentario proporcionado por el Evaluador
 *         UsuarioId:
 *           type: number
 *           description: Trabajador que fue Evaluado
 *       example:
 *         id: "1"
 *         FechaEvaluacion: "2025-04-11"
 *         Evaluador: "Anibal Jhamil Huaman Verastein"
 *         Puntaje: "97"
 *         Comentario: "Excelente Trabajador"
 *         UsuarioId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Evaluacion
 *   description: API para gestionar Evaluacions
 */
/**
 * @swagger
 * /rrhh/evaluacion:
 *   get:
 *     summary: Obtiene la lista de Evaluacions
 *     tags: [Evaluacion]
 *     responses:
 *       200:
 *         description: Lista de Evaluacions.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Evaluacion'
 */
router.get("/", verifyToken, evaluacionController.getEvaluacionAll);
/**
 * @swagger
 * /rrhh/evaluacion/{id}:
 *   get:
 *     summary: Obtiene un Evaluacion por ID
 *     tags: [Evaluacion]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Evaluacion
 *     responses:
 *       200:
 *         description: Evaluacion encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Evaluacion'
 *       404:
 *         description: Evaluacion no encontrado.
 */
router.get("/:id", verifyToken, evaluacionController.getEvaluacionById);
/**
 * @swagger
 * /rrhh/evaluacion/trabajador/{trabajador}:
 *   get:
 *     summary: Obtiene un Evaluaciones por trabajador
 *     tags: [Evaluacion]
 *     parameters:
 *       - in: path
 *         name: trabajador
 *         schema:
 *           type: number
 *         required: true
 *         description: Evaluaciones del trabajador
 *     responses:
 *       200:
 *         description: Evaluaciones encontradas.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Evaluacion'
 *       404:
 *         description: Evaluacion no encontrado.
 */
router.get("/trabajador/:trabajador", verifyToken, evaluacionController.getEvaluacionByTrabajador);
/**
 * @swagger
 * /rrhh/Evaluacion/create:
 *   post:
 *     summary: Crea un nuevo Evaluacion
 *     tags: [Evaluacion]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Evaluacion'
 *     responses:
 *       201:
 *         description: Evaluacion creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Evaluacion'
 */
router.post("/create", verifyToken, evaluacionController.createEvaluacion);
/**
 * @swagger
 * /rrhh/evaluacion/update/{id}:
 *   put:
 *     summary: Actualiza un Evaluacion existente
 *     tags: [Evaluacion]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Evaluacion a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Evaluacion'
 *     responses:
 *       200:
 *         description: Evaluacion actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Evaluacion'
 *       404:
 *         description: Evaluacion no encontrado.
 */
router.put("/update/:id", verifyToken, evaluacionController.updateEvaluacion);
/**
 * @swagger
 * /rrhh/evaluacion/{id}:
 *   delete:
 *     summary: Elimina un Evaluacion por ID
 *     tags: [Evaluacion]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Evaluacion a eliminar
 *     responses:
 *       200:
 *         description: Evaluacion eliminado exitosamente.
 *       404:
 *         description: Evaluacion no encontrado.
 */
router.delete("/:id", evaluacionController.deleteEvaluacion);

// Rutas de filtrado individual


module.exports = router;
