const express = require("express");
const EvolucionController = require("../controllers/evolucion.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware
// const { validatePerson } = require("../middlewares/validation.middleware");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Evolucion:
 *       type: object
 *       required:
 *         - FechaEvolucion
 *         - Descripcion
 *         - ProcedimientoId
 *         - CodigoCIE10P
 *         - DiagnosticoCIE10P
 *         - CodigoCIE10D
 *         - DiagnosticoCIE10D
 *         - AgendamientoId
 *         - PacienteId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Evolucion
 *         FechaEvolucion:
 *           type: Date
 *           description: Nombre del Evolucion
 *         Descripcion:
 *           type: string
 *           description: Descripcion de la Evolucion
 *         ProcedimientoId:
 *           type: string
 *           description: Identificador de Procedimiento
 *         CodigoCIE10P:
 *           type: string
 *           description: Codigo CIE10 Presuntivo
 *         DiagnosticoCIE10P:
 *           type: string
 *           description: Diagnostico Presuntivo de la Evolucion
 *         CodigoCIE10D:
 *           type: number
 *           description: Codigo CIE10 Definitivo
 *         DiagnosticoCIE10D:
 *           type: string
 *           description: Diagnostico Definitivo de la Evolucion
 *         AgendamientoId:
 *           type: number
 *           description: Identificador de Agendamiento
 *         PacienteId:
 *           type: number
 *           description: Identificador de la Historia Clinica
 *       example:
 *         id: "1"
 *         FechaEvolucion: "2025-11-11"
 *         Descripcion: "gfdsgsg"
 *         ProcedimientoId: "1"
 *         CodigoCIE10P: "gds32"
 *         DiagnosticoCIE10P: "gsdfgfd gfdgdfgdf gfdsgsdf"
 *         CodigoCIE10D: "gsdf23"
 *         DiagnosticoCIE10D: "gfsdg dfgdfsdsgssfdgsdfs fdsgdfsg"
 *         AgendamientoId: "1"
 *         PacienteId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Evolucion
 *   description: API para gestionar Evolucion
 */
/**
 * @swagger
 * /paciente/evolucion:
 *   get:
 *     summary: Obtiene la lista de Evolucion
 *     tags: [Evolucion]
 *     responses:
 *       200:
 *         description: Lista de Evolucion.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Evolucion'
 */
router.get("/", verifyToken, EvolucionController.getEvolucionAll);
/**
 * @swagger
 * /paciente/evolucion/{id}:
 *   get:
 *     summary: Obtiene un Evolucion por ID
 *     tags: [Evolucion]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Evolucion
 *     responses:
 *       200:
 *         description: Evolucion encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Evolucion'
 *       404:
 *         description: Evolucion no encontrado.
 */
router.get("/:id", verifyToken, EvolucionController.getEvolucionById);
/**
 * @swagger
 * /paciente/evolucion/create:
 *   post:
 *     summary: Crea un nuevo Evolucion
 *     tags: [Evolucion]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Evolucion'
 *     responses:
 *       201:
 *         description: Evolucion creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Evolucion'
 */
router.post("/create", verifyToken, EvolucionController.createEvolucion); // Aplicamos validación
/**
 * @swagger
 * /paciente/evolucion/update/{id}:
 *   put:
 *     summary: Actualiza un Evolucion existente
 *     tags: [Evolucion]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Evolucion a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Evolucion'
 *     responses:
 *       200:
 *         description: Evolucion actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Evolucion'
 *       404:
 *         description: Evolucion no encontrado.
 */
router.put("/update/:id", verifyToken, EvolucionController.updateEvolucion);
/**
 * @swagger
 * /paciente/evolucion/{id}:
 *   delete:
 *     summary: Elimina un Evolucion por ID
 *     tags: [Evolucion]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Evolucion a eliminar
 *     responses:
 *       200:
 *         description: Evolucion eliminado exitosamente.
 *       404:
 *         description: Evolucion no encontrado.
 */
router.delete("/:id", verifyToken, EvolucionController.deleteEvolucion);
// 🔍 Rutas individuales para filtrado
/**
 * @swagger
 * /paciente/evolucion/filtrar/paciente/{paciente}:
 *   get:
 *     summary: Obtiene un Evolucion por paciente
 *     tags: [Evolucion]
 *     parameters:
 *       - in: path
 *         name: paciente
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del paciente
 *     responses:
 *       200:
 *         description: Evolucion encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Evolucion'
 *       404:
 *         description: Evolucion no encontrado.
 */
// ruta para filtrar las citas del paciente
router.get("/filtrar/paciente/:paciente", verifyToken, EvolucionController.getEvolucionByPaciente);

module.exports = router;