const express = require("express");
const ExploracionFisicaController = require("../controllers/exploracionfisica.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware
// const { validatePerson } = require("../middlewares/validation.middleware");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     ExploracionFisica:
 *       type: object
 *       required:
 *         - FrecuenciaCardiaca
 *         - FrecuenciaRespiratoria
 *         - PresionArterial
 *         - Temperatura
 *         - SaturaciondeOxigeno
 *         - ExamenClinicoGeneral
 *         - OdontoEstomatologia
 *         - AgendamientoId
 *         - PacienteId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del ExploracionFisica
 *         FrecuenciaCardiaca:
 *           type: string
 *           description: Frecuencia Cardiaca
 *         FrecuenciaRespiratoria:
 *           type: string
 *           description: Frecuencia Respiratoria
 *         PresionArterial:
 *           type: string
 *           description: Presion Arterial
 *         Temperatura:
 *           type: string
 *           description: Temperatura
 *         SaturaciondeOxigeno:
 *           type: string
 *           description: Saturacion de Oxigeno
 *         ExamenClinicoGeneral:
 *           type: number
 *           description: Examen Clinico General
 *         OdontoEstomatologia:
 *           type: string
 *           description: Odonto Estomatologia
 *         AgendamientoId:
 *           type: number
 *           description: Identificador de Agendamiento
 *         PacienteId:
 *           type: number
 *           description: Identificador de la Historia Clinica
 *       example:
 *         id: "1"
 *         FrecuenciaCardiaca: "fdasfsd"
 *         FrecuenciaRespiratoria: "fdsagsdag"
 *         PresionArterial: "gdsagad"
 *         Temperatura: "gdsag"
 *         SaturaciondeOxigeno: "gadsgs"
 *         ExamenClinicoGeneral: "gadsgdg"
 *         OdontoEstomatologia: "gsdaggd"
 *         AgendamientoId: "1"
 *         PacienteId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: ExploracionFisica
 *   description: API para gestionar ExploracionFisica
 */
/**
 * @swagger
 * /paciente/exploracionfisica:
 *   get:
 *     summary: Obtiene la lista de ExploracionFisica
 *     tags: [ExploracionFisica]
 *     responses:
 *       200:
 *         description: Lista de ExploracionFisica.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/ExploracionFisica'
 */
router.get("/", verifyToken, ExploracionFisicaController.getExploracionFisicaAll);
/**
 * @swagger
 * /paciente/exploracionfisica/{id}:
 *   get:
 *     summary: Obtiene un ExploracionFisica por ID
 *     tags: [ExploracionFisica]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del ExploracionFisica
 *     responses:
 *       200:
 *         description: ExploracionFisica encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ExploracionFisica'
 *       404:
 *         description: ExploracionFisica no encontrado.
 */
router.get("/:id", verifyToken, ExploracionFisicaController.getExploracionFisicaById);
/**
 * @swagger
 * /paciente/exploracionfisica/create:
 *   post:
 *     summary: Crea un nuevo ExploracionFisica
 *     tags: [ExploracionFisica]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ExploracionFisica'
 *     responses:
 *       201:
 *         description: ExploracionFisica creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ExploracionFisica'
 */
router.post("/create", verifyToken, ExploracionFisicaController.createExploracionFisica); // Aplicamos validación
/**
 * @swagger
 * /paciente/exploracionfisica/update/{id}:
 *   put:
 *     summary: Actualiza un ExploracionFisica existente
 *     tags: [ExploracionFisica]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del ExploracionFisica a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ExploracionFisica'
 *     responses:
 *       200:
 *         description: ExploracionFisica actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ExploracionFisica'
 *       404:
 *         description: ExploracionFisica no encontrado.
 */
router.put("/update/:id", verifyToken, ExploracionFisicaController.updateExploracionFisica);
/**
 * @swagger
 * /paciente/exploracionfisica/{id}:
 *   delete:
 *     summary: Elimina un ExploracionFisica por ID
 *     tags: [ExploracionFisica]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del ExploracionFisica a eliminar
 *     responses:
 *       200:
 *         description: ExploracionFisica eliminado exitosamente.
 *       404:
 *         description: ExploracionFisica no encontrado.
 */
router.delete("/:id", verifyToken, ExploracionFisicaController.deleteExploracionFisica);
// 🔍 Rutas individuales para filtrado
/**
 * @swagger
 * /paciente/exploracionfisica/filtrar/paciente/{paciente}:
 *   get:
 *     summary: Obtiene un ExploracionFisica por paciente
 *     tags: [ExploracionFisica]
 *     parameters:
 *       - in: path
 *         name: paciente
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del paciente
 *     responses:
 *       200:
 *         description: ExploracionFisica encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ExploracionFisica'
 *       404:
 *         description: ExploracionFisica no encontrado.
 */
// ruta para filtrar las citas del paciente
router.get("/filtrar/paciente/:paciente", verifyToken, ExploracionFisicaController.getExploracionFisicaByPaciente);

module.exports = router;