const express = require("express");
const FacturaController = require("../controllers/factura.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Factura:
 *       type: object
 *       required:
 *         - FechaFacturacion
 *         - NroOperacion
 *         - TotalCopagoFijo
 *         - ImporteTotal
 *         - CopagoVariable
 *         - CopagoTotal
 *         - EmpresaId
 *         - PacienteId
 *         - IAFAId
 *         - IngresoId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único de la Factura
 *         FechaFacturacion:
 *           type: Date
 *           description: Fecha Facturacion de la Factura
 *         NroOperacion:
 *           type: string
 *           description: NroOperacion de la Factura
 *         TotalCopagoFijo:
 *           type: number
 *           description: Total Copago Fijo de la Factura
 *         ImporteTotal:
 *           type: number
 *           description: Importe Total de la Factura
 *         CopagoVariable:
 *           type: number
 *           description: Copago Variable de la Factura
 *         CopagoTotal:
 *           type: number
 *           description: Copago Total de la Factura
 *         EmpresaId:
 *           type: number
 *           description: Identifiador del Empresa
 *         PacienteId:
 *           type: number
 *           description: Identifiador del v
 *         IAFAId:
 *           type: number
 *           description: Identifiador de la IAFA
 *         IngresoId:
 *           type: number
 *           description: Identifiador del Ingreso
 *       example:
 *         id: "1"
 *         FechaFacturacion: "Calle Cantuta 124"
 *         NroOperacion: "Dos cuadras antes de Unias, frente a la ferreteria Paola"
 *         TotalCopagoFijo: "243252"
 *         ImporteTotal: "Peru"
 *         CopagoVariable: "Junin"
 *         CopagoTotal: "Huancayo"
 *         EmpresaId: "Huancayo"
 *         PacienteId: "-3.4324325325"
 *         IAFAId: "84.4324325235325"
 *         IngresoId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Factura
 *   description: API para gestionar Factura
 */
/**
 * @swagger
 * /economia/factura:
 *   get:
 *     summary: Obtiene la lista de Factura
 *     tags: [Factura]
 *     responses:
 *       200:
 *         description: Lista de Factura.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Factura'
 */
router.get("/", verifyToken, FacturaController.getFacturaAll);
/**
 * @swagger
 * /economia/factura/{id}:
 *   get:
 *     summary: Obtiene un Factura por ID
 *     tags: [Factura]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID de la Factura
 *     responses:
 *       200:
 *         description: Factura encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Factura'
 *       404:
 *         description: Factura no encontrado.
 */
router.get("/:id", verifyToken, FacturaController.getFacturaById);
/**
 * @swagger
 * /economia/factura/create:
 *   post:
 *     summary: Crea un nuevo Factura
 *     tags: [Factura]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Factura'
 *     responses:
 *       201:
 *         description: Factura creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Factura'
 */
router.post("/create", verifyToken, FacturaController.createFactura);
/**
 * @swagger
 * /economia/factura/update/{id}:
 *   put:
 *     summary: Actualiza un Factura existente
 *     tags: [Factura]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID de la Factura a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Factura'
 *     responses:
 *       200:
 *         description: Factura actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Factura'
 *       404:
 *         description: Factura no encontrado.
 */
router.put("/update/:id", verifyToken, FacturaController.updateFactura);
/**
 * @swagger
 * /economia/factura/{id}:
 *   delete:
 *     summary: Elimina un Factura por ID
 *     tags: [Factura]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Factura a eliminar
 *     responses:
 *       200:
 *         description: Factura eliminado exitosamente.
 *       404:
 *         description: Factura no encontrado.
 */
router.delete("/:id", verifyToken, FacturaController.deleteFactura);
/**
 * @swagger
 * /economia/factura/filtrar/sede/{sede}:
 *   get:
 *     summary: Obtiene un Factura por usuario
 *     tags: [Factura]
 *     parameters:
 *       - in: path
 *         name: sede
 *         schema:
 *           type: number
 *         required: true
 *         description: sede de la Factura
 *     responses:
 *       200:
 *         description: Factura encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Factura'
 *       404:
 *         description: Factura no encontrado.
 */
router.get("/filtrar/sede/:sede", verifyToken, FacturaController.filtrarPorSede);
/**
 * @swagger
 * /economia/factura/filtrar/paciente/{paciente}:
 *   get:
 *     summary: Obtiene un Factura por usuario
 *     tags: [Factura]
 *     parameters:
 *       - in: path
 *         name: paciente
 *         schema:
 *           type: number
 *         required: true
 *         description: paciente de la Factura
 *     responses:
 *       200:
 *         description: Factura encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Factura'
 *       404:
 *         description: Factura no encontrado.
 */
router.get("/filtrar/paciente/:paciente", verifyToken, FacturaController.filtrarPorPaciente);

module.exports = router;
