const express = require("express");
const FacturacionController = require("../controllers/facturacion.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Facturacion:
 *       type: object
 *       required:
 *         - Detalle
 *         - Monto
 *         - IGV
 *         - Total
 *         - CoverturaId
 *         - FacturaId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Facturacion
 *         Detalle:
 *           type: string
 *           description: Detalle de la Facturacion
 *         Monto:
 *           type: number
 *           description: Monto de la Facturacion
 *         IGV:
 *           type: number
 *           description: IGV de la Facturacion
 *         Total:
 *           type: number
 *           description: Total de la Facturacion
 *         CoverturaId:
 *           type: number
 *           description: Identifiador de la Covertura
 *         FacturaId:
 *           type: number
 *           description: Identifiador de la Factura
 *       example:
 *         id: "1"
 *         Detalle: "Carie Moderada"
 *         Monto: "100"
 *         IGV: "18"
 *         Total: "118"
 *         CoverturaId: "1"
 *         FacturaId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Facturacion
 *   description: API para gestionar Facturacion
 */
/**
 * @swagger
 * /economia/facturacion:
 *   get:
 *     summary: Obtiene la lista de Facturacion
 *     tags: [Facturacion]
 *     responses:
 *       200:
 *         description: Lista de Facturacion.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Facturacion'
 */
router.get("/", verifyToken, FacturacionController.getFacturacionAll);
/**
 * @swagger
 * /economia/facturacion/{id}:
 *   get:
 *     summary: Obtiene un Facturacion por ID
 *     tags: [Facturacion]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Facturacion
 *     responses:
 *       200:
 *         description: Facturacion encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Facturacion'
 *       404:
 *         description: Facturacion no encontrado.
 */
router.get("/:id", verifyToken, FacturacionController.getFacturacionById);
/**
 * @swagger
 * /economia/facturacion/create:
 *   post:
 *     summary: Crea un nuevo Facturacion
 *     tags: [Facturacion]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Facturacion'
 *     responses:
 *       201:
 *         description: Facturacion creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Facturacion'
 */
router.post("/create", verifyToken, FacturacionController.createFacturacion);
/**
 * @swagger
 * /economia/facturacion/update/{id}:
 *   put:
 *     summary: Actualiza un Facturacion existente
 *     tags: [Facturacion]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Facturacion a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Facturacion'
 *     responses:
 *       200:
 *         description: Facturacion actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Facturacion'
 *       404:
 *         description: Facturacion no encontrado.
 */
router.put("/update/:id", verifyToken, FacturacionController.updateFacturacion);
/**
 * @swagger
 * /economia/facturacion/{id}:
 *   delete:
 *     summary: Elimina un Facturacion por ID
 *     tags: [Facturacion]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Facturacion a eliminar
 *     responses:
 *       200:
 *         description: Facturacion eliminado exitosamente.
 *       404:
 *         description: Facturacion no encontrado.
 */
router.delete("/:id", verifyToken, FacturacionController.deleteFacturacion);
/**
 * @swagger
 * /economia/facturacion/filtrar/factura/{factura}:
 *   get:
 *     summary: Obtiene un Facturacion por factura
 *     tags: [Facturacion]
 *     parameters:
 *       - in: path
 *         name: factura
 *         schema:
 *           type: number
 *         required: true
 *         description: factura del Facturacion
 *     responses:
 *       200:
 *         description: Facturacion encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Facturacion'
 *       404:
 *         description: Facturacion no encontrado.
 */
router.get("/filtrar/factura/:factura", verifyToken, FacturacionController.getFacturacionByFactura);

module.exports = router;
