const express = require("express");
const FotosController = require("../controllers/fotos.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware
// const { validatePerson } = require("../middlewares/validation.middleware");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Fotos:
 *       type: object
 *       required:
 *         - RutaImagen
 *         - TipoFoto
 *         - PacienteId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Fotos
 *         RutaImagen:
 *           type: string
 *           description: Ruta de la Imagen
 *         TipoFoto:
 *           type: string
 *           description: Tipo Foto
 *         PacienteId:
 *           type: number
 *           description: Identificador de Historia Clinica
 *       example:
 *         id: "1"
 *         RutaImagen: "fdsfsaf/fds/sad "
 *         TipoFoto: "Frontal"
 *         PacienteId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Fotos
 *   description: API para gestionar Fotos
 */
/**
 * @swagger
 * /paciente/fotos:
 *   get:
 *     summary: Obtiene la lista de Fotos
 *     tags: [Fotos]
 *     responses:
 *       200:
 *         description: Lista de Fotos.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Fotos'
 */
router.get("/", verifyToken, FotosController.getFotosAll);
/**
 * @swagger
 * /paciente/fotos/{id}:
 *   get:
 *     summary: Obtiene un Fotos por ID
 *     tags: [Fotos]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Fotos
 *     responses:
 *       200:
 *         description: Fotos encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Fotos'
 *       404:
 *         description: Fotos no encontrado.
 */
router.get("/:id", verifyToken, FotosController.getFotosById);
/**
 * @swagger
 * /paciente/fotos/create:
 *   post:
 *     summary: Crea un nuevo Fotos
 *     tags: [Fotos]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Fotos'
 *     responses:
 *       201:
 *         description: Fotos creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Fotos'
 */
router.post("/create", verifyToken, FotosController.createFotos); // Aplicamos validación
/**
 * @swagger
 * /paciente/fotos/update/{id}:
 *   put:
 *     summary: Actualiza un Fotos existente
 *     tags: [Fotos]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Fotos a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Fotos'
 *     responses:
 *       200:
 *         description: Fotos actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Fotos'
 *       404:
 *         description: Fotos no encontrado.
 */
router.put("/update/:id", verifyToken, FotosController.updateFotos);
/**
 * @swagger
 * /paciente/fotos/{id}:
 *   delete:
 *     summary: Elimina un Fotos por ID
 *     tags: [Fotos]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Fotos a eliminar
 *     responses:
 *       200:
 *         description: Fotos eliminado exitosamente.
 *       404:
 *         description: Fotos no encontrado.
 */
router.delete("/:id", verifyToken, FotosController.deleteFotos);
// 🔍 Rutas individuales para filtrado
/**
 * @swagger
 * /paciente/fotos/filtrar/paciente/{paciente}:
 *   get:
 *     summary: Obtiene un Fotos por paciente
 *     tags: [Fotos]
 *     parameters:
 *       - in: path
 *         name: paciente
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del paciente
 *     responses:
 *       200:
 *         description: Fotos encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Fotos'
 *       404:
 *         description: Fotos no encontrado.
 */
// ruta para filtrar las citas del paciente
router.get("/filtrar/paciente/:paciente", verifyToken, FotosController.getFotosByPaciente);

module.exports = router;