const express = require("express");
const historialaboralController = require("../controllers/historialaboral.controller");
const { verifyToken } = require('../middlewares/auth.middleware');

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     HistorialLaboral:
 *       type: object
 *       required:
 *         - FechaInicio
 *         - FechaFin
 *         - Cargo
 *         - Observacion
 *         - UsuarioId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único de la HistorialLaboral
 *         FechaInicio:
 *           type: Date
 *           description: Fecha Inicio de Contrato
 *         FechaFin:
 *           type: Date
 *           description: Fecha Fin de Contrato
 *         Cargo:
 *           type: string
 *           description: Cargo en el Contrato
 *         Observacion:
 *           type: string
 *           description: Observaciones en el Contrato
 *         UsuarioId:
 *           type: number
 *           description: El Contrato le Pertenece al Trabajador
 *       example:
 *         id: "1"
 *         FechaInicio: "2023-06-01"
 *         FechaFin: "2024-04-31"
 *         Cargo: "Tecnico Asistencial"
 *         Observacion: "Contrato por menos de un anio evitando  ..."
 *         UsuarioId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: HistorialLaboral
 *   description: API para gestionar HistorialLaborals
 */
/**
 * @swagger
 * /rrhh/historiallaboral:
 *   get:
 *     summary: Obtiene la lista de HistorialLaborals
 *     tags: [HistorialLaboral]
 *     responses:
 *       200:
 *         description: Lista de HistorialLaborals.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/HistorialLaboral'
 */
router.get("/", verifyToken, historialaboralController.getHistoriaLaboralAll);
/**
 * @swagger
 * /rrhh/historiallaboral/trabajador/{trabajador}:
 *   get:
 *     summary: Obtiene un HistorialLaboral por trabajador
 *     tags: [HistorialLaboral]
 *     parameters:
 *       - in: path
 *         name: trabajador
 *         schema:
 *           type: number
 *         required: true
 *         description: Historial Laboral del trabajador
 *     responses:
 *       200:
 *         description: Historial Laboral encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/HistorialLaboral'
 *       404:
 *         description: HistorialLaboral no encontrado.
 */
router.get("/trabajador/:trabajador", verifyToken, historialaboralController.getHistoriaLaboralByTrabajador);
/**
 * @swagger
 * /rrhh/historiallaboral/{id}:
 *   get:
 *     summary: Obtiene un HistorialLaboral por ID
 *     tags: [HistorialLaboral]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del HistorialLaboral
 *     responses:
 *       200:
 *         description: HistorialLaboral encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/HistorialLaboral'
 *       404:
 *         description: HistorialLaboral no encontrado.
 */
router.get("/:id", verifyToken, historialaboralController.getHistoriaLaboralById);
/**
 * @swagger
 * /rrhh/historiallaboral/create:
 *   post:
 *     summary: Crea un nuevo HistorialLaboral
 *     tags: [HistorialLaboral]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/HistorialLaboral'
 *     responses:
 *       201:
 *         description: HistorialLaboral creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/HistorialLaboral'
 */
router.post("/create", verifyToken, historialaboralController.createHistoriaLaboral);
/**
 * @swagger
 * /rrhh/historiallaboral/update/{id}:
 *   put:
 *     summary: Actualiza un HistorialLaboral existente
 *     tags: [HistorialLaboral]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del HistorialLaboral a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/HistorialLaboral'
 *     responses:
 *       200:
 *         description: HistorialLaboral actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/HistorialLaboral'
 *       404:
 *         description: HistorialLaboral no encontrado.
 */
router.put("/update/:id", verifyToken, historialaboralController.updateHistoriaLaboral);
/**
 * @swagger
 * /rrhh/historiallaboral/{id}:
 *   delete:
 *     summary: Elimina un HistorialLaboral por ID
 *     tags: [HistorialLaboral]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del HistorialLaboral a eliminar
 *     responses:
 *       200:
 *         description: HistorialLaboral eliminado exitosamente.
 *       404:
 *         description: HistorialLaboral no encontrado.
 */
router.delete("/:id", historialaboralController.deleteHistoriaLaboral);

// Rutas de filtrado individual


module.exports = router;
