const express = require("express");
const iafaController = require("../controllers/iafa.controller");
const { verifyToken } = require('../middlewares/auth.middleware');

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Iafa:
 *       type: object
 *       required:
 *         - RazonSocial
 *         - RUC
 *         - Direccion
 *         - Contacto
 *         - FechaCacelacion
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único de la IAFA
 *         RazonSocial:
 *           type: string
 *           description: RazonSocial de la IAFA
 *         RUC:
 *           type: string
 *           description: RUC de la IAFA
 *         Direccion:
 *           type: string
 *           description: Direccion de la IAFA
 *         Contacto:
 *           type: string
 *           description: Contacto de la IAFA
 *         FechaCacelacion:
 *           type: string
 *           description: FechaC acelacion de la IAFA
 *       example:
 *         id: "1"
 *         Nombre: "Administrador🔑"
 */
/**
 * @swagger
 * tags:
 *   name: Iafa
 *   description: API para gestionar IAFAs
 */
/**
 * @swagger
 * /generales/iafa:
 *   get:
 *     summary: Obtiene la lista de IAFAs
 *     tags: [Iafa]
 *     responses:
 *       200:
 *         description: Lista de IAFAs.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Iafa'
 */
router.get(
    "/", verifyToken,
    iafaController.getIafaAll
);
/**
 * @swagger
 * /generales/iafa/{id}:
 *   get:
 *     summary: Obtiene un IAFA por ID
 *     tags: [Iafa]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del IAFA
 *     responses:
 *       200:
 *         description: IAFA encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Iafa'
 *       404:
 *         description: IAFA no encontrado.
 */
router.get(
    "/:id", verifyToken,
    iafaController.getIafaById
);
/**
 * @swagger
 * /generales/iafa/ruc/{ruc}:
 *   get:
 *     summary: Obtiene un IAFA por ruc
 *     tags: [Iafa]
 *     parameters:
 *       - in: path
 *         name: ruc
 *         schema:
 *           type: number
 *         required: true
 *         description: ruc del IAFA
 *     responses:
 *       200:
 *         description: IAFA encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Iafa'
 *       404:
 *         description: IAFA no encontrado.
 */
router.get(
    "/ruc/:ruc", verifyToken,
    iafaController.getIafaByRUC
);
/**
 * @swagger
 * /generales/iafa/create:
 *   post:
 *     summary: Crea un nuevo IAFA
 *     tags: [Iafa]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Iafa'
 *     responses:
 *       201:
 *         description: IAFA creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Iafa'
 */
router.post(
    "/create", verifyToken,
    iafaController.createIafa
);
/**
 * @swagger
 * /generales/iafa/update/{id}:
 *   put:
 *     summary: Actualiza un IAFA existente
 *     tags: [Iafa]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del IAFA a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Iafa'
 *     responses:
 *       200:
 *         description: IAFA actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Iafa'
 *       404:
 *         description: IAFA no encontrado.
 */
router.put(
    "/update/:id", verifyToken,
    iafaController.updateIafa
);
/**
 * @swagger
 * /generales/iafa/{id}:
 *   delete:
 *     summary: Elimina un IAFA por ID
 *     tags: [Iafa]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del IAFA a eliminar
 *     responses:
 *       200:
 *         description: IAFA eliminado exitosamente.
 *       404:
 *         description: IAFA no encontrado.
 */
router.delete(
    "/:id",
    iafaController.deleteIafa
);

module.exports = router;
