const express = require("express");
const IncidenciaController = require("../controllers/incidencia.controller");
const { verifyToken } = require('../middlewares/auth.middleware');

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Incidencia:
 *       type: object
 *       required:
 *         - Codigo
 *         - Descripcion
 *         - FechaIncidente
 *         - FechaRegistro
 *         - DOCPersonal
 *         - Personal
 *         - SedeId
 *         - EstadoId
 *         - TipoIncidenciaId
 *         - ClienteId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único de Incidencia
 *         Codigo:
 *           type: string
 *           description: Codigo de la Incidencia (se autogenera con datos recabados de los involucrados)
 *         Descripcion:
 *           type: string
 *           description: Descripcion de la Incidencia
 *         FechaIncidente:
 *           type: Date
 *           description: Fecha en la que ocurrio en incidente
 *         FechaRegistro:
 *           type: Date
 *           description: Fecha en la que se registra el incidente
 *         DOCPersonal:
 *           type: string
 *           description: Documento del Personal
 *         Personal:
 *           type: string
 *           description: Nombres y Apellidos del Personal
 *         SedeId:
 *           type: number
 *           description: Sede asociada a la Incidencia
 *         EstadoId:
 *           type: number
 *           description: Estado Incidencia (Nuevo, En Revisión, En Proceso, Pendiente, Resuelto, Cerrado y Reabierto)
 *         TipoIncidenciaId:
 *           type: number
 *           description: Tipo de Incidencia (Reclamo, Queja, Sugerencia, Felicitación)
 *         ClienteId:
 *           type: number
 *           description: Cliente asociado a la Incidencia
 *       example:
 *         id: "1"
 *         Codigo: "AJHV-I-AJHV-123"
 *         Descripcion: "Ejemplo de Incidencia"
 *         FechaIncidente: "2025-04-11"
 *         FechaRegistro: "2025-04-12"
 *         DOCPersonal: "72547410"
 *         Personal: "Anibal Jhamil Huaman Verastein"
 *         SedeId: "1"
 *         EstadoId: "1"
 *         TipoIncidenciaId: "1"
 *         ClienteId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Incidencia
 *   description: API para gestionar Incidencias
 */
/**
 * @swagger
 * /incidencias/incidencia:
 *   get:
 *     summary: Obtiene la lista de Incidencias
 *     tags: [Incidencia]
 *     responses:
 *       200:
 *         description: Lista de Incidencias.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Incidencia'
 */
router.get("/", verifyToken, IncidenciaController.getIncidenciaAll);
/**
 * @swagger
 * /incidencias/incidencia/filter/sede/{sede}:
 *   get:
 *     summary: Obtiene Incidencias asociadas a una sede
 *     tags: [Incidencia]
 *     parameters:
 *       - in: path
 *         name: sede
 *         schema:
 *           type: number
 *         required: true
 *         description: sede de la Incidencia
 *     responses:
 *       200:
 *         description: Incidencias encontradas de la sede asociada.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Incidencia'
 *       404:
 *         description: Incidencias no encontrado por la Estado Escogida.
 */
router.get("/filter/sede/:sede", verifyToken, IncidenciaController.getIncidenciaBySede);
/**
 * @swagger
 * /incidencias/incidencia/filter/sedeytipo/{sede}/{tipo}:
 *   get:
 *     summary: Obtiene Incidencias asociadas a una Sede y Estado
 *     tags: [Incidencia]
 *     parameters:
 *       - in: path
 *         name: sede
 *         schema:
 *           type: number
 *         required: true
 *         description: sede de la Incidencia
 *       - in: path
 *         name: tipo
 *         schema:
 *           type: number
 *         required: true
 *         description: tipo de la Incidencia
 *     responses:
 *       200:
 *         description: Incidencias encontradas de la Sede y tipo asociada.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Incidencia'
 *       404:
 *         description: Incidencias no encontrado por la Estado Escogida.
 */
router.get("/filter/sedeytipo/:sede/:tipo", verifyToken, IncidenciaController.getIncidenciaBySedeyTipo);
/**
 * @swagger
 * /incidencias/incidencia/filter/sedetipoyestado/{sede}/{tipo}/{estado}:
 *   get:
 *     summary: Obtiene Incidencias asociadas a una Sede y Estado
 *     tags: [Incidencia]
 *     parameters:
 *       - in: path
 *         name: sede
 *         schema:
 *           type: number
 *         required: true
 *         description: sede de la Incidencia
 *       - in: path
 *         name: tipo
 *         schema:
 *           type: number
 *         required: true
 *         description: tipo de la Incidencia
 *       - in: path
 *         name: estado
 *         schema:
 *           type: number
 *         required: true
 *         description: Estado de la Incidencia
 *     responses:
 *       200:
 *         description: Incidencias encontradas de la Sede y Estado asociada.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Incidencia'
 *       404:
 *         description: Incidencias no encontrado por la Estado Escogida.
 */
router.get("/filter/sedetipoyestado/:sede/:tipo/:estado", verifyToken, IncidenciaController.getIncidenciaBySedeTipoyEstado);
/**
 * @swagger
 * /incidencias/incidencia/filter/personal/{personal}:
 *   get:
 *     summary: Obtiene Incidencias asociadas a un personal
 *     tags: [Incidencia]
 *     parameters:
 *       - in: path
 *         name: personal
 *         schema:
 *           type: string
 *         required: true
 *         description: documento del personal
 *     responses:
 *       200:
 *         description: Incidencias encontradas de la Estado asociada.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Incidencia'
 *       404:
 *         description: Incidencias no encontrado por la Estado Escogida.
 */
router.get("/filter/personal/:personal", verifyToken, IncidenciaController.getIncidenciaByPersonal);
/**
 * @swagger
 * /incidencias/incidencia/filter/codigo/{codigo}:
 *   get:
 *     summary: Obtiene Incidencia asociada a un codigo
 *     tags: [Incidencia]
 *     parameters:
 *       - in: path
 *         name: codigo
 *         schema:
 *           type: string
 *         required: true
 *         description: Codigo de la Incidencia
 *     responses:
 *       200:
 *         description: Incidencia encontrada por codigo asociada.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Incidencia'
 *       404:
 *         description: Incidencias no encontrado por la Estado Escogida.
 */
router.get("/buscar/codigo/:codigo", verifyToken, IncidenciaController.getIncidenciaByCodigo);
/**
 * @swagger
 * /incidencias/incidencia/{id}:
 *   get:
 *     summary: Obtiene un Incidencia por ID
 *     tags: [Incidencia]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Incidencia
 *     responses:
 *       200:
 *         description: Incidencia encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Incidencia'
 *       404:
 *         description: Incidencia no encontrado.
 */
router.get("/:id", verifyToken, IncidenciaController.getIncidenciaById);
/**
 * @swagger
 * /incidencias/incidencia/create:
 *   post:
 *     summary: Crea un nuevo Incidencia
 *     tags: [Incidencia]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Incidencia'
 *     responses:
 *       201:
 *         description: Incidencia creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Incidencia'
 */
router.post("/create", verifyToken, IncidenciaController.createIncidencia);
/**
 * @swagger
 * /incidencias/incidencia/update/{id}:
 *   put:
 *     summary: Actualiza un Incidencia existente
 *     tags: [Incidencia]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Incidencia a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Incidencia'
 *     responses:
 *       200:
 *         description: Incidencia actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Incidencia'
 *       404:
 *         description: Incidencia no encontrado.
 */
router.put("/update/:id", verifyToken, IncidenciaController.updateIncidencia);
/**
 * @swagger
 * /incidencias/incidencia/{id}:
 *   delete:
 *     summary: Elimina un Incidencia por ID
 *     tags: [Incidencia]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Incidencia a eliminar
 *     responses:
 *       200:
 *         description: Incidencia eliminado exitosamente.
 *       404:
 *         description: Incidencia no encontrado.
 */
router.delete("/:id", IncidenciaController.deleteIncidencia);

module.exports = router;
