const express = require("express");
const IncidenciaDetalleController = require("../controllers/incidenciadetalle.controller");
const { verifyToken } = require('../middlewares/auth.middleware');

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     IncidenciaDetalle:
 *       type: object
 *       required:
 *         - QuePaso
 *         - QuienProboco
 *         - CuandoPaso
 *         - DondePaso
 *         - Causa
 *         - Responsable
 *         - Observaciones
 *         - Seguimiento
 *         - FechaCumplimiento
 *         - Verificacion
 *         - Procede
 *         - AccionaTomar
 *         - IncidenciaId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del usuario
 *         QuePaso:
 *           type: string
 *           description: Analisis de incidencia - que paso?
 *         QuienProboco:
 *           type: string
 *           description: Analisis de incidencia - quien proboco?
 *         CuandoPaso:
 *           type: string
 *           description: Analisis de incidencia - cuando paso?
 *         DondePaso:
 *           type: string
 *           description: Analisis de incidencia - donde paso?
 *         Causa:
 *           type: string
 *           description: Causa del incidente      
 *         Responsable:
 *           type: string
 *           description: Responsable de analizar en incidente
 *         Observaciones:
 *           type: string
 *           description: Observacion u objeciones frente al inicidente
 *         Seguimiento:
 *           type: string
 *           description: Seguimiento de la incidencia
 *         FechaCumplimiento:
 *           type: string
 *           description: Fecha de finalizacion
 *         Verificacion:
 *           type: string
 *           description: Informacion sobre la verificacion del sistema
 *         Procede:
 *           type: string
 *           description: Toma de desicion si procede tomar acciones
 *         AccionaTomar:
 *           type: string
 *           description: Accion que se tomo
 *         IncidenciaId:
 *           type: number
 *           description: A que Incidencia pertenece
 *       example:
 *         id: "1"
 *         QuePaso: "Que paso en la incidencia"
 *         QuienProboco: "Quien provoco la incidencia"
 *         CuandoPaso: "Cuando paso la incidencia"
 *         DondePaso: "Donde paso la incidencia"
 *         Causa: "Causa de la incidencia"
 *         Responsable: "Responsable de la incidencia"
 *         Observaciones: "Observaciones de la incidencia"
 *         Seguimiento: "Seguimiento de la incidencia"
 *         FechaCumplimiento: "2025-10-10"
 *         Verificacion: "Verificacion de la incidencia"
 *         Procede: "Si Procede"
 *         AccionaTomar: "Accion a tomar"
 *         IncidenciaId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: IncidenciaDetalle
 *   description: API para gestionar IncidenciaDetalles
 */
/**
 * @swagger
 * /incidencias/incidenciadetalle:
 *   get:
 *     summary: Obtiene la lista de IncidenciaDetalles
 *     tags: [IncidenciaDetalle]
 *     responses:
 *       200:
 *         description: Lista de IncidenciaDetalles.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/IncidenciaDetalle'
 */
router.get("/", verifyToken, IncidenciaDetalleController.getIncidenciaDetalleAll);
/**
 * @swagger
 * /incidencias/incidenciadetalle/incidencia/{incidencia}:
 *   get:
 *     summary: Obtiene IncidenciaDetalles asociadas a una Incidencia
 *     tags: [IncidenciaDetalle]
 *     parameters:
 *       - in: path
 *         name: incidencia
 *         schema:
 *           type: number
 *         required: true
 *         description: ID de la Incidencia
 *     responses:
 *       200:
 *         description: IncidenciaDetalles encontradas de la Incidencia asociada.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/IncidenciaDetalle'
 *       404:
 *         description: IncidenciaDetalles no encontrado por la Incidencia Escogida.
 */
router.get("/incidencia/:incidencia", verifyToken, IncidenciaDetalleController.getIncidenciaDetalleByIncidencia);
/**
 * @swagger
 * /incidencias/incidenciadetalle/{id}:
 *   get:
 *     summary: Obtiene un IncidenciaDetalle por ID
 *     tags: [IncidenciaDetalle]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del IncidenciaDetalle
 *     responses:
 *       200:
 *         description: IncidenciaDetalle encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/IncidenciaDetalle'
 *       404:
 *         description: IncidenciaDetalle no encontrado.
 */
router.get("/:id", verifyToken, IncidenciaDetalleController.getIncidenciaDetalleById);
/**
 * @swagger
 * /incidencias/incidenciadetalle/create:
 *   post:
 *     summary: Crea un nuevo IncidenciaDetalle
 *     tags: [IncidenciaDetalle]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/IncidenciaDetalle'
 *     responses:
 *       201:
 *         description: IncidenciaDetalle creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/IncidenciaDetalle'
 */
router.post("/create", verifyToken, IncidenciaDetalleController.createIncidenciaDetalle);
/**
 * @swagger
 * /incidencias/incidenciadetalle/update/{id}:
 *   put:
 *     summary: Actualiza un IncidenciaDetalle existente
 *     tags: [IncidenciaDetalle]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del IncidenciaDetalle a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/IncidenciaDetalle'
 *     responses:
 *       200:
 *         description: IncidenciaDetalle actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/IncidenciaDetalle'
 *       404:
 *         description: IncidenciaDetalle no encontrado.
 */
router.put("/update/:id", verifyToken, IncidenciaDetalleController.updateIncidenciaDetalle);
/**
 * @swagger
 * /incidencias/IncidenciaDetalle/{id}:
 *   delete:
 *     summary: Elimina un IncidenciaDetalle por ID
 *     tags: [IncidenciaDetalle]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del IncidenciaDetalle a eliminar
 *     responses:
 *       200:
 *         description: IncidenciaDetalle eliminado exitosamente.
 *       404:
 *         description: IncidenciaDetalle no encontrado.
 */
router.delete("/:id", verifyToken, IncidenciaDetalleController.deleteIncidenciaDetalle);

module.exports = router;
