const express = require("express");
const IngresoController = require("../controllers/ingreso.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Ingreso:
 *       type: object
 *       required:
 *         - FechaRegistro
 *         - Personal
 *         - Odontologo
 *         - DatoProveedor
 *         - MontoSoles
 *         - MontoDolarMomento
 *         - MontoaPagar
 *         - MontoPagado
 *         - MontoRestante
 *         - EstadoPagoId
 *         - TratamientoId
 *         - SedeId
 *         - TipoPagoId
 *         - TipoIngresoId
 *         - UsuarioId
 *         - TipoProveedorId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Ingreso
 *         FechaRegistro:
 *           type: Date
 *           description: Fecha Registro del Ingreso
 *         Personal:
 *           type: string
 *           description: Personal del Ingreso
 *         Odontologo:
 *           type: string
 *           description: Odontologo del Ingreso
 *         DatoProveedor:
 *           type: string
 *           description: DatoProveedor del Ingreso
 *         MontoSoles:
 *           type: number
 *           description: MontoSoles del Ingreso
 *         MontoDolarMomento:
 *           type: number
 *           description: MontoDolarMomento del Ingreso
 *         MontoaPagar:
 *           type: number
 *           description: Monto a Pagar del Ingreso
 *         MontoPagado:
 *           type: number
 *           description: Monto Pagado del Ingreso
 *         MontoRestante:
 *           type: number
 *           description: Monto Restante del Ingreso
 *         EstadoPagoId:
 *           type: number
 *           description: Estado Pago del Ingreso
 *         TratamientoId:
 *           type: number
 *           description: Tratamiento del Ingreso
 *         SedeId:
 *           type: number
 *           description: Sede del Ingreso
 *         TipoPagoId:
 *           type: number
 *           description: Tipo Pago del Ingreso
 *         TipoIngresoId:
 *           type: number
 *           description: Tipo Ingreso del Ingreso
 *         UsuarioId:
 *           type: number
 *           description: Usuario del Ingreso
 *         TipoProveedorId:
 *           type: number
 *           description: Tipo Proveedor del Ingreso
 *       example:
 *         id: "1"
 *         FechaRegistro: "2025-12-11"
 *         Personal: "Personal que registra el pago"
 *         Odontologo: "Si es por atencion datos del odontologo que atendio"
 *         DatoProveedor: "Nombre Persona que pago"
 *         MontoSoles: "100"
 *         MontoDolarMomento: "30"
 *         MontoaPagar: "100"
 *         MontoPagado: "100"
 *         MontoRestante: "0"
 *         EstadoPagoId: "1"
 *         TratamientoId: "1"
 *         SedeId: "1"
 *         TipoPagoId: "1"
 *         TipoIngresoId: "1"
 *         UsuarioId: "1"
 *         TipoProveedorId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Ingreso
 *   description: API para gestionar Ingreso
 */
/**
 * @swagger
 * /economia/ingreso:
 *   get:
 *     summary: Obtiene la lista de Ingreso
 *     tags: [Ingreso]
 *     responses:
 *       200:
 *         description: Lista de Ingreso.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Ingreso'
 */
router.get("/", verifyToken, IngresoController.getIngresoAll);
/**
 * @swagger
 * /economia/ingreso/{id}:
 *   get:
 *     summary: Obtiene un Ingreso por ID
 *     tags: [Ingreso]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Ingreso
 *     responses:
 *       200:
 *         description: Ingreso encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Ingreso'
 *       404:
 *         description: Ingreso no encontrado.
 */
router.get("/:id", verifyToken, IngresoController.getIngresoById);
/**
 * @swagger
 * /economia/ingreso/create:
 *   post:
 *     summary: Crea un nuevo Ingreso
 *     tags: [Ingreso]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Ingreso'
 *     responses:
 *       201:
 *         description: Ingreso creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Ingreso'
 */
router.post("/create", verifyToken, IngresoController.createIngreso);
/**
 * @swagger
 * /economia/ingreso/update/{id}:
 *   put:
 *     summary: Actualiza un Ingreso existente
 *     tags: [Ingreso]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Ingreso a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Ingreso'
 *     responses:
 *       200:
 *         description: Ingreso actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Ingreso'
 *       404:
 *         description: Ingreso no encontrado.
 */
router.put("/update/:id", verifyToken, IngresoController.updateIngreso);
/**
 * @swagger
 * /economia/ingreso/{id}:
 *   delete:
 *     summary: Elimina un Ingreso por ID
 *     tags: [Ingreso]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Ingreso a eliminar
 *     responses:
 *       200:
 *         description: Ingreso eliminado exitosamente.
 *       404:
 *         description: Ingreso no encontrado.
 */
router.delete("/:id", verifyToken, IngresoController.deleteIngreso);
/**
 * @swagger
 * /economia/ingreso/filtrar/sede/{sede}:
 *   get:
 *     summary: Obtiene un Ingreso por sede
 *     tags: [Ingreso]
 *     parameters:
 *       - in: path
 *         name: sede
 *         schema:
 *           type: number
 *         required: true
 *         description: sede del Ingreso
 *     responses:
 *       200:
 *         description: Ingreso encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Ingreso'
 *       404:
 *         description: Ingreso no encontrado.
 */
router.get("/filtrar/sede/:sede", verifyToken, IngresoController.getIngresosBySede);
/**
 * @swagger
 * /economia/ingreso/filtrar/sedeytipopago/{sede}/{tipopago}:
 *   get:
 *     summary: Obtiene un Ingreso por sede y tipo pago
 *     tags: [Ingreso]
 *     parameters:
 *       - in: path
 *         name: sede
 *         schema:
 *           type: number
 *         required: true
 *         description: sede del Ingreso
 *       - in: path
 *         name: tipopago
 *         schema:
 *           type: number
 *         required: true
 *         description: tipopago del Ingreso
 *     responses:
 *       200:
 *         description: Ingreso encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Ingreso'
 *       404:
 *         description: Ingreso no encontrado.
 */
router.get("/filtrar/sedeytipopago/:sede/:tipopago", verifyToken, IngresoController.getIngresoBySedeandTipoPago);
/**
 * @swagger
 * /economia/ingreso/filtrar/sedeytipoingreso/{sede}/{tipoingreso}:
 *   get:
 *     summary: Obtiene un Ingreso por sede
 *     tags: [Ingreso]
 *     parameters:
 *       - in: path
 *         name: sede
 *         schema:
 *           type: number
 *         required: true
 *         description: sede del Ingreso
 *       - in: path
 *         name: tipoingreso
 *         schema:
 *           type: number
 *         required: true
 *         description: tipo ingreso del Ingreso
 *     responses:
 *       200:
 *         description: Ingreso encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Ingreso'
 *       404:
 *         description: Ingreso no encontrado.
 */
router.get("/filtrar/sedeytipoingreso/:sede/:tipoingreso", verifyToken, IngresoController.getIngresoBySedeandTipoIngreso);
/**
 * @swagger
 * /economia/ingreso/filtrar/sedeytipoproveedor/{sede}/{tipoproveedor}:
 *   get:
 *     summary: Obtiene un Ingreso por sede
 *     tags: [Ingreso]
 *     parameters:
 *       - in: path
 *         name: sede
 *         schema:
 *           type: number
 *         required: true
 *         description: sede del Ingreso
 *       - in: path
 *         name: tipoproveedor
 *         schema:
 *           type: number
 *         required: true
 *         description: tipo proveedor del Ingreso
 *     responses:
 *       200:
 *         description: Ingreso encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Ingreso'
 *       404:
 *         description: Ingreso no encontrado.
 */
router.get("/filtrar/sedeytipoproveedor/:sede/:tipoproveedor", verifyToken, IngresoController.getIngresoBySedeandTipoProveedor);
/**
 * @swagger
 * /economia/ingreso/reporte/global/{fechainicio}/{fechafin}:
 *   get:
 *     summary: Obtiene un Ingreso por sede
 *     tags: [Ingreso]
 *     parameters:
 *       - in: path
 *         name: fechainicio
 *         schema:
 *           type: Date
 *         required: true
 *         description: fecha inicio del Ingreso
 *       - in: path
 *         name: fechafin
 *         schema:
 *           type: Date
 *         required: true
 *         description: fecha fin del Ingreso
 *     responses:
 *       200:
 *         description: Ingreso encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Ingreso'
 *       404:
 *         description: Ingreso no encontrado.
 */
router.get("/reporte/global/:fechainicio/:fechafin", verifyToken, IngresoController.getIngresoByReport);


module.exports = router;
