const express = require("express");
const { verifyToken } = require('../middlewares/auth.middleware');
const insumoController = require("../controllers/insumo.controller");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Insumo:
 *       type: object
 *       required:
 *         - Nombre
 *         - Descripcion
 *         - UnidaddeMedida
 *         - StockMinimo
 *         - StockMaximo
 *         - SubCategoriaInventarioId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único de la Insumo
 *         Nombre:
 *           type: string
 *           description: Nombre del Insumo
 *         Descripcion:
 *           type: string
 *           description: Descripcion del Insumo
 *         UnidaddeMedida:
 *           type: string
 *           description: Unidad de Medida para contar los insumos
 *         StockMinimo:
 *           type: number
 *           description: Stock Minimo para elevar alerta
 *         StockMaximo:
 *           type: number
 *           description: Stock Maximo
 *         SubCategoriaInventarioId:
 *           type: number
 *           description: Sub Categoria a la que pertenece
 *       example:
 *         id: "1"
 *         Nombre: "Espejo dental"
 *         Descripcion: "examinar la cavidad bucal"
 *         UnidaddeMedida: "Unidad"
 *         StockMinimo: "2"
 *         StockMaximo: "12"
 *         SubCategoriaInventarioId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Insumo
 *   description: API para gestionar Insumos
 */
/**
 * @swagger
 * /inventario/insumo:
 *   get:
 *     summary: Obtiene la lista de Insumos
 *     tags: [Insumo]
 *     responses:
 *       200:
 *         description: Lista de Insumos.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Insumo'
 */
router.get("/", verifyToken, insumoController.getInsumoAll);
/**
 * @swagger
 * /inventario/insumo/categoria/{categoria}:
 *   get:
 *     summary: Obtiene un Insumo por categoria
 *     tags: [Insumo]
 *     parameters:
 *       - in: path
 *         name: categoria
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Insumo
 *     responses:
 *       200:
 *         description: Insumos encontrados por categoria.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Insumo'
 *       404:
 *         description: Insumo no encontrado.
 */
router.get("/categoria/:categoria", verifyToken, insumoController.getInsumoByCategoria);
/**
 * @swagger
 * /inventario/insumo/subcategoria/{subcategoria}:
 *   get:
 *     summary: Obtiene un Insumo por subcategoria
 *     tags: [Insumo]
 *     parameters:
 *       - in: path
 *         name: subcategoria
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Insumo
 *     responses:
 *       200:
 *         description: Insumos encontrados por subcategoria.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Insumo'
 *       404:
 *         description: Insumo no encontrado.
 */
router.get("/subcategoria/:subcategoria", verifyToken, insumoController.getInsumoBySubCategoria);
/**
 * @swagger
 * /inventario/insumo/{id}:
 *   get:
 *     summary: Obtiene un Insumo por ID
 *     tags: [Insumo]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Insumo
 *     responses:
 *       200:
 *         description: Insumo encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Insumo'
 *       404:
 *         description: Insumo no encontrado.
 */
router.get("/:id", verifyToken, insumoController.getInsumoById);
/**
 * @swagger
 * /inventario/insumo/create:
 *   post:
 *     summary: Crea un nuevo Insumo
 *     tags: [Insumo]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Insumo'
 *     responses:
 *       201:
 *         description: Insumo creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Insumo'
 */
router.post("/create", verifyToken, insumoController.createInsumo);
/**
 * @swagger
 * /inventario/insumo/update/{id}:
 *   put:
 *     summary: Actualiza un Insumo existente
 *     tags: [Insumo]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Insumo a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Insumo'
 *     responses:
 *       200:
 *         description: Insumo actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Insumo'
 *       404:
 *         description: Insumo no encontrado.
 */
router.put("/update/:id", verifyToken, insumoController.updateInsumo);
/**
 * @swagger
 * /inventario/insumo/{id}:
 *   delete:
 *     summary: Elimina un Insumo por ID
 *     tags: [Insumo]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Insumo a eliminar
 *     responses:
 *       200:
 *         description: Insumo eliminado exitosamente.
 *       404:
 *         description: Insumo no encontrado.
 */
router.delete("/:id", verifyToken, insumoController.deleteInsumo);

// Rutas de filtrado individual


module.exports = router;
