const express = require("express");
const { verifyToken } = require('../middlewares/auth.middleware');
const inventarioController = require("../controllers/inventario.controller");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Inventario:
 *       type: object
 *       required:
 *         - Cantidad
 *         - Estado
 *         - AlmacenId
 *         - InsumoId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único de la Inventario
 *         Cantidad:
 *           type: number
 *           description: Cantidad segun el insumo que pertenece a tal subalmacen
 *         Estado:
 *           type: string
 *           description: Stokeo del Insumo (stockeado, acabado, alerta)
 *         AlmacenId:
 *           type: number
 *           description: almacen al que pertenece
 *         InsumoId:
 *           type: number
 *           description: Insumo del que se cuenta
 *       example:
 *         id: "1"
 *         Cantidad: "4"
 *         Estado: "stockeado"
 *         AlmacenId: "1"
 *         InsumoId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Inventario
 *   description: API para gestionar Inventarios
 */
/**
 * @swagger
 * /inventario/inventario:
 *   get:
 *     summary: Obtiene la lista de Inventarios
 *     tags: [Inventario]
 *     responses:
 *       200:
 *         description: Lista de Inventarios.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Inventario'
 */
router.get("/", verifyToken, inventarioController.getInventarioAll);
/**
 * @swagger
 * /inventario/inventario/sede/{sede}:
 *   get:
 *     summary: Obtiene un Inventario por almacen
 *     tags: [Inventario]
 *     parameters:
 *       - in: path
 *         name: sede
 *         schema:
 *           type: number
 *         required: true
 *         description: sede a la que pertenece el Inventario
 *     responses:
 *       200:
 *         description: Inventario encontrado por sede
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Inventario'
 *       404:
 *         description: Inventario no encontrado.
 */
router.get("/sede/:sede", verifyToken, inventarioController.getInventarioBySede);
/**
 * @swagger
 * /inventario/inventario/almacen/{almacen}:
 *   get:
 *     summary: Obtiene un Inventario por almacen
 *     tags: [Inventario]
 *     parameters:
 *       - in: path
 *         name: almacen
 *         schema:
 *           type: number
 *         required: true
 *         description: Almacen de un Inventario
 *     responses:
 *       200:
 *         description: Inventario encontrado por Almacen
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Inventario'
 *       404:
 *         description: Inventario no encontrado.
 */
router.get("/almacen/:almacen", verifyToken, inventarioController.getInventarioByAlmacen);
/**
 * @swagger
 * /inventario/inventario/insumo/{insumo}:
 *   get:
 *     summary: Obtiene un Inventario por insumo
 *     tags: [Inventario]
 *     parameters:
 *       - in: path
 *         name: insumo
 *         schema:
 *           type: number
 *         required: true
 *         description: insumo de un Inventario
 *     responses:
 *       200:
 *         description: Inventario encontrado por insumo
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Inventario'
 *       404:
 *         description: Inventario no encontrado.
 */
router.get("/sedeeinsumo/:sede/:insumo", verifyToken, inventarioController.getInventarioBySedeeInsumo);
/**
 * @swagger
 * /inventario/inventario/almaceneinsumo/{almacen}/{insumo}:
 *   get:
 *     summary: Obtiene un Inventario por ID
 *     tags: [Inventario]
 *     parameters:
 *       - in: path
 *         name: almacen
 *         schema:
 *           type: number
 *         required: true
 *         description: Almacen del Inventario
 *       - in: path
 *         name: insumo
 *         schema:
 *           type: number
 *         required: true
 *         description: Insumo del Inventario
 *     responses:
 *       200:
 *         description: Inventario encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Inventario'
 *       404:
 *         description: Inventario no encontrado.
 */
router.get("/almaceneinsumo/:almacen/:insumo", verifyToken, inventarioController.getInventarioByAlmacenANDInsumo);
/**
 * @swagger
 * /inventario/inventario/{id}:
 *   get:
 *     summary: Obtiene un Inventario por ID
 *     tags: [Inventario]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Inventario
 *     responses:
 *       200:
 *         description: Inventario encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Inventario'
 *       404:
 *         description: Inventario no encontrado.
 */
router.get("/:id", verifyToken, inventarioController.getInventarioById);
/**
 * @swagger
 * /inventario/inventario/create:
 *   post:
 *     summary: Crea un nuevo Inventario
 *     tags: [Inventario]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Inventario'
 *     responses:
 *       201:
 *         description: Inventario creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Inventario'
 */
router.post("/create", verifyToken, inventarioController.createInventario);
/**
 * @swagger
 * /inventario/inventario/update/{id}:
 *   put:
 *     summary: Actualiza un Inventario existente
 *     tags: [Inventario]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Inventario a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Inventario'
 *     responses:
 *       200:
 *         description: Inventario actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Inventario'
 *       404:
 *         description: Inventario no encontrado.
 */
router.put("/update/:id", verifyToken, inventarioController.updateInventario);
/**
 * @swagger
 * /inventario/inventario/{id}:
 *   delete:
 *     summary: Elimina un Inventario por ID
 *     tags: [Inventario]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Inventario a eliminar
 *     responses:
 *       200:
 *         description: Inventario eliminado exitosamente.
 *       404:
 *         description: Inventario no encontrado.
 */
router.delete("/:id", verifyToken, inventarioController.deleteInventario);

// Rutas de filtrado individual


module.exports = router;
