const express = require("express");
const ItemLaboratorioController = require("../controllers/itemlaboratorio.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     ItemLaboratorio:
 *       type: object
 *       required:
 *         - Nombre
 *         - LaboratorioId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del ItemLaboratorio
 *         Nombre:
 *           type: string
 *           description: Nombre del Item Laboratorio
 *         LaboratorioId:
 *           type: number
 *           description: Relacion del ItemLaboratorio con el Laboratorio
 *       example:
 *         id: "1"
 *         Nombre: "Hemograma Completo"
 *         LaboratorioId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: ItemLaboratorio
 *   description: API para gestionar ItemLaboratorio
 */
/**
 * @swagger
 * /laboratorio/item:
 *   get:
 *     summary: Obtiene la lista de ItemLaboratorio
 *     tags: [ItemLaboratorio]
 *     responses:
 *       200:
 *         description: Lista de ItemLaboratorio.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/ItemLaboratorio'
 */
router.get("/", verifyToken, ItemLaboratorioController.getItemLaboratorioAll);
/**
 * @swagger
 * /laboratorio/item/{id}:
 *   get:
 *     summary: Obtiene un ItemLaboratorio por ID
 *     tags: [ItemLaboratorio]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del ItemLaboratorio
 *     responses:
 *       200:
 *         description: ItemLaboratorio encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ItemLaboratorio'
 *       404:
 *         description: ItemLaboratorio no encontrado.
 */
router.get("/:id", verifyToken, ItemLaboratorioController.getItemLaboratorioById);
/**
 * @swagger
 * /laboratorio/item/create:
 *   post:
 *     summary: Crea un nuevo ItemLaboratorio
 *     tags: [ItemLaboratorio]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ItemLaboratorio'
 *     responses:
 *       201:
 *         description: ItemLaboratorio creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ItemLaboratorio'
 */
router.post("/create", verifyToken, ItemLaboratorioController.createItemLaboratorio);
/**
 * @swagger
 * /laboratorio/item/update/{id}:
 *   put:
 *     summary: Actualiza un ItemLaboratorio existente
 *     tags: [ItemLaboratorio]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del ItemLaboratorio a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ItemLaboratorio'
 *     responses:
 *       200:
 *         description: ItemLaboratorio actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ItemLaboratorio'
 *       404:
 *         description: ItemLaboratorio no encontrado.
 */
router.put("/update/:id", verifyToken, ItemLaboratorioController.updateItemLaboratorio);
/**
 * @swagger
 * /laboratorio/item/{id}:
 *   delete:
 *     summary: Elimina un ItemLaboratorio por ID
 *     tags: [ItemLaboratorio]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del ItemLaboratorio a eliminar
 *     responses:
 *       200:
 *         description: ItemLaboratorio eliminado exitosamente.
 *       404:
 *         description: ItemLaboratorio no encontrado.
 */
router.delete("/:id", verifyToken, ItemLaboratorioController.deleteItemLaboratorio);
/**
 * @swagger
 * /laboratorio/item/filtro/laboratorio/{laboratorio}:
 *   get:
 *     summary: Obtiene un ItemLaboratorio por laboratorio
 *     tags: [ItemLaboratorio]
 *     parameters:
 *       - in: path
 *         name: laboratorio
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del laboratorio
 *     responses:
 *       200:
 *         description: ItemLaboratorioes del laboratorio encontrados.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ItemLaboratorio'
 *       404:
 *         description: ItemLaboratorio no encontrado.
 */
// 🔍 Rutas individuales para búsqueda
router.get("/filtro/laboratorio/:laboratorio", verifyToken, ItemLaboratorioController.getItemLaboratorioByLaboratorio);

module.exports = router;
